<?php

namespace Drupal\viewer\Plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Component\Transliteration\TransliterationInterface;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileUrlGeneratorInterface;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\viewer\Entity\Viewer;
use Drupal\viewer\Entity\ViewerSourceInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * ViewerBase plugin base class.
 */
class ViewerBase extends PluginBase implements ViewerInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;
  use DependencySerializationTrait;

  /**
   * Viewer entity.
   *
   * @var \Drupal\viewer\Entity\Viewer|null
   */
  protected ?Viewer $viewer = NULL;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * The transliteration service.
   *
   * @var \Drupal\Component\Transliteration\TransliterationInterface
   */
  protected TransliterationInterface $transliteration;

  /**
   * The viewer processor plugin manager.
   *
   * @var \Drupal\viewer\Plugin\ViewerProcessorManager
   */
  protected ViewerProcessorManager $viewerProcessorManager;

  /**
   * The viewer cell plugin manager.
   *
   * @var \Drupal\viewer\Plugin\ViewerCellManager
   */
  protected ViewerCellManager $viewerCellManager;

  /**
   * The file URL generator.
   *
   * @var \Drupal\Core\File\FileUrlGeneratorInterface
   */
  protected FileUrlGeneratorInterface $fileUrlGenerator;

  /**
   * The form builder service.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected FormBuilderInterface $formBuilder;

  /**
   * Constructs a ViewerBase plugin.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\Component\Transliteration\TransliterationInterface $transliteration
   *   The transliteration service.
   * @param \Drupal\viewer\Plugin\ViewerProcessorManager $viewer_processor_manager
   *   The viewer processor plugin manager.
   * @param \Drupal\viewer\Plugin\ViewerCellManager $viewer_cell_manager
   *   The viewer cell plugin manager.
   * @param \Drupal\Core\File\FileUrlGeneratorInterface $file_url_generator
   *   The file URL generator.
   * @param \Drupal\Core\Form\FormBuilderInterface $form_builder
   *   The form builder service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    TransliterationInterface $transliteration,
    ViewerProcessorManager $viewer_processor_manager,
    ViewerCellManager $viewer_cell_manager,
    FileUrlGeneratorInterface $file_url_generator,
    FormBuilderInterface $form_builder,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
    $this->transliteration = $transliteration;
    $this->viewerProcessorManager = $viewer_processor_manager;
    $this->viewerCellManager = $viewer_cell_manager;
    $this->fileUrlGenerator = $file_url_generator;
    $this->formBuilder = $form_builder;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('transliteration'),
      $container->get('plugin.manager.viewer_processor'),
      $container->get('plugin.manager.viewer_cell'),
      $container->get('file_url_generator'),
      $container->get('form_builder'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function setViewer(Viewer $viewer): static {
    $this->viewer = $viewer;
    return $this;
  }

  /**
   * {@inheritdoc}
   */
  public function getViewer(): ?Viewer {
    return $this->viewer;
  }

  /**
   * Gets the Viewer source entity.
   *
   * @return \Drupal\viewer\Entity\ViewerSourceInterface|null
   *   The viewer source entity.
   */
  public function getViewerSource(): ?ViewerSourceInterface {
    return $this->viewer?->getViewerSource();
  }

  /**
   * {@inheritdoc}
   */
  public function getName(): TranslatableMarkup|string|null {
    return $this->pluginDefinition['name'] ?? NULL;
  }

  /**
   * Check if viewer source is empty.
   *
   * @return bool
   *   TRUE if empty viewer source, FALSE otherwise.
   */
  public function isEmptyViewerSource(): bool {
    return !isset($this->pluginDefinition['empty_viewer_source']) ? FALSE : (bool) $this->pluginDefinition['empty_viewer_source'];
  }

  /**
   * Get viewer types.
   *
   * @return array
   *   Array of viewer types.
   */
  public function viewerTypes(): array {
    return !empty($this->pluginDefinition['viewer_types']) ? $this->pluginDefinition['viewer_types'] : [];
  }

  /**
   * Get the processor plugin.
   *
   * @return \Drupal\viewer\Plugin\ViewerProcessorInterface|null
   *   The processor plugin or NULL.
   */
  public function getProcessPlugin(): ?ViewerProcessorInterface {
    if (!empty($this->pluginDefinition['processor'])) {
      return $this->viewerProcessorManager->createInstance($this->pluginDefinition['processor']);
    }
    return NULL;
  }

  /**
   * Check if filterable.
   *
   * @return bool
   *   TRUE if filterable, FALSE otherwise.
   */
  public function filterable(): bool {
    return !empty($this->pluginDefinition['filters']);
  }

  /**
   * Check if requirements are met.
   *
   * @return bool
   *   TRUE if requirements are met.
   */
  public function requirementsAreMet(): bool {
    return TRUE;
  }

  /**
   * Get the response data.
   *
   * @return array
   *   The response data.
   */
  public function getResponse(): array {
    return $this->getViewer()->getDataAsArray(TRUE);
  }

  /**
   * Get renderable array.
   *
   * @return array
   *   The renderable array.
   */
  public function getRenderable(): array {
    return [
      '#markup' => 'This is your plugin default template',
    ];
  }

  /**
   * Build settings form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $params
   *   Additional parameters.
   *
   * @return array
   *   The form array.
   */
  public function settingsForm(array $form, FormStateInterface $form_state, array $params = []): array {
    $settings = $params['settings'];
    $form['plugin'] = [
      '#type' => 'vertical_tabs',
    ];
    $form['header'] = [
      '#type' => 'details',
      '#title' => $this->t('Header'),
      '#group' => 'plugin',
      '#weight' => -9,
    ];
    $form['header']['title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Title'),
      '#default_value' => !empty($settings['title']) ? $settings['title'] : '',
    ];
    $form['header']['subtitle'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Subtitle'),
      '#default_value' => !empty($settings['subtitle']) ? $settings['subtitle'] : '',
    ];
    $form['header']['header_summary'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Summary'),
      '#default_value' => !empty($settings['header_summary']) ? $settings['header_summary'] : '',
    ];
    $form['footer'] = [
      '#type' => 'details',
      '#title' => $this->t('Footer'),
      '#group' => 'plugin',
      '#weight' => -8,
    ];
    $form['footer']['footer_summary'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Summary'),
      '#default_value' => !empty($settings['footer_summary']) ? $settings['footer_summary'] : '',
    ];
    return $form;
  }

  /**
   * Get settings values from form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The settings values.
   */
  public function settingsValues(array &$form, FormStateInterface $form_state): array {
    return [
      'title' => $form_state->getValue('title'),
      'subtitle' => $form_state->getValue('subtitle'),
      'header_summary' => $form_state->getValue('header_summary'),
      'footer_summary' => $form_state->getValue('footer_summary'),
    ];
  }

  /**
   * Build configuration form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param array $params
   *   Additional parameters.
   *
   * @return array
   *   The form array.
   */
  public function configurationForm(array $form, FormStateInterface $form_state, array $params = []): array {
    return $form;
  }

  /**
   * Get configuration values from form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The configuration values.
   */
  public function configurationValues(array &$form, FormStateInterface $form_state): array {
    return [];
  }

  /**
   * Load Viewer uuid.
   *
   * @return string|null
   *   The viewer UUID or NULL.
   */
  public function getId(): ?string {
    return $this->getViewer()?->uuid();
  }

  /**
   * Load Viewer settings.
   *
   * @return array
   *   The viewer settings.
   */
  public function getSettings(): array {
    return $this->getViewer()->getSettings();
  }

  /**
   * Load Viewer filters.
   *
   * @return array
   *   The viewer filters.
   */
  public function getFilters(): array {
    return $this->getViewer()->getFilters();
  }

  /**
   * Load Viewer configuration (transformer).
   *
   * @return array
   *   The viewer configuration.
   */
  public function getConfiguration(): array {
    return $this->getViewer()->getConfiguration();
  }

  /**
   * Get the file URL generator service.
   *
   * @return \Drupal\Core\File\FileUrlGeneratorInterface
   *   The file URL generator.
   */
  protected function getFileUrlGenerator(): FileUrlGeneratorInterface {
    return $this->fileUrlGenerator;
  }

  /**
   * Generates a machine name from a string.
   *
   * @param string $string
   *   The string to convert.
   *
   * @return string
   *   The machine name.
   */
  protected function getMachineName(string $string): string {
    $transliterated = $this->transliteration->transliterate($string, LanguageInterface::LANGCODE_DEFAULT, '_');
    $transliterated = mb_strtolower($transliterated);
    $transliterated = preg_replace('@[^a-z0-9_.]+@', '_', $transliterated);
    return $transliterated;
  }

  /**
   * Get all date formats.
   *
   * @return array
   *   Array of date formats.
   */
  protected function getDateFormats(): array {
    $formats = [];
    $entities = $this->entityTypeManager->getStorage('date_format')->loadMultiple();
    foreach ($entities as $format) {
      $formats[$format->id()] = $format->label();
    }
    return $formats;
  }

  /**
   * Get cell plugins.
   *
   * @return array
   *   Array of cell plugins.
   */
  protected function getCellPlugins(): array {
    $plugins = [];
    $plugins['as_is'] = $this->t('As is');
    foreach ($this->viewerCellManager->getDefinitions() as $id => $plugin) {
      $plugin = $this->viewerCellManager->createInstance($plugin['id']);
      if (!empty($plugin->getApplicableViewers()) && in_array($this->getPluginId(), $plugin->getApplicableViewers())) {
        $plugins[$id] = $plugin->getName();
      }
    }
    return $plugins;
  }

}
