<?php

namespace Drupal\viewer\Plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\Site\Settings;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\TempStore\PrivateTempStore;
use Drupal\Core\TempStore\PrivateTempStoreFactory;
use Drupal\file\Entity\File;
use Drupal\file\FileInterface;
use Drupal\viewer\Traits\PluginBatchTrait;
use Drupal\viewer\Traits\TempKeyValTrait;
use Drupal\viewer\ViewerInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\Mime\MimeTypes;

/**
 * ViewerSourceBase plugin base class.
 */
class ViewerSourceBase extends PluginBase implements ViewerSourceInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;
  use DependencySerializationTrait;
  use PluginBatchTrait;
  use TempKeyValTrait;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Private temporary store factory.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStoreFactory
   */
  protected PrivateTempStoreFactory $tempStoreFactory;

  /**
   * Stores private data for viewer source multistep.
   *
   * @var \Drupal\Core\TempStore\PrivateTempStore
   */
  protected PrivateTempStore $store;

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected EntityTypeManagerInterface $entityTypeManager;

  /**
   * Constructs a ViewerSourceBase plugin.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   * @param \Drupal\Core\TempStore\PrivateTempStoreFactory $temp_store_factory
   *   The private temp store factory.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, FileSystemInterface $file_system, PrivateTempStoreFactory $temp_store_factory, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->fileSystem = $file_system;
    $this->tempStoreFactory = $temp_store_factory;
    $this->store = $this->tempStoreFactory->get('viewer_source_multistep_values');
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static($configuration, $plugin_id, $plugin_definition,
      $container->get('file_system'),
      $container->get('tempstore.private'),
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getName(): TranslatableMarkup|string|null {
    return $this->pluginDefinition['name'] ?? NULL;
  }

  /**
   * Check if this source supports cron.
   *
   * @return bool
   *   TRUE if cron is supported.
   */
  public function isCron(): bool {
    return !empty($this->pluginDefinition['cron']);
  }

  /**
   * Method to check requirements for the source plugin.
   *
   * @return bool
   *   TRUE if requirements are met.
   */
  public function requirementsAreMet(): bool {
    return TRUE;
  }

  /**
   * {@inheritdoc}
   */
  public function sourceForm(array $form, FormStateInterface $form_state, mixed $viewer_type, mixed $viewer_source): array {
    $properties_form = $viewer_type->propertiesForm([]);
    if ($properties_form !== $form) {
      $form = array_merge($form, $properties_form);
    }
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitSourceForm(array &$form, FormStateInterface $form_state, mixed $viewer_type): array|false {
    return FALSE;
  }

  /**
   * {@inheritdoc}
   */
  public function importForm(array $form, FormStateInterface $form_state, mixed $viewer_type, mixed $viewer_source): array {
    $form = $this->sourceForm($form, $form_state, $viewer_source->getTypePlugin(), $viewer_source);
    if ($properties_form = $viewer_type->propertiesForm([])) {
      foreach (array_keys($properties_form) as $element) {
        unset($form[$element]);
      }
    }
    unset($form['import_frequency'], $form['credentials']);
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitImportForm(array &$form, FormStateInterface $form_state, mixed $viewer_source = NULL): array|false {
    return FALSE;
  }

  /**
   * Build settings form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param mixed $viewer_source
   *   The viewer source entity.
   * @param array $settings
   *   The settings array.
   *
   * @return array
   *   The form array.
   */
  public function settingsForm(array $form, FormStateInterface $form_state, mixed $viewer_source = NULL, array $settings = []): array {
    $properties_form = $viewer_source->getTypePlugin()->propertiesForm($settings);
    if ($properties_form !== $form) {
      $form = array_merge($form, $properties_form);
    }
    return $form;
  }

  /**
   * Submit settings form.
   *
   * @param array $form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   * @param mixed $viewer_source
   *   The viewer source entity.
   * @param array $settings
   *   The settings array.
   *
   * @return array
   *   The settings array.
   */
  public function submitSettingsForm(array &$form, FormStateInterface $form_state, mixed $viewer_source = NULL, array $settings = []): array {
    return $viewer_source->getTypePlugin()->submitPropertiesForm($form_state);
  }

  /**
   * Get file from path.
   *
   * @param string $filepath
   *   The file path.
   * @param array $mime_types
   *   Array of allowed mime types.
   *
   * @return \Drupal\file\FileInterface|null
   *   The file entity or NULL.
   */
  public function getFileFromPath(string $filepath, array $mime_types): ?FileInterface {
    // Only allowing file types specified in the ViewerType plugin.
    $mime_type = (new MimeTypes())->guessMimeType($filepath);
    if (in_array($mime_type, $mime_types)) {
      $handle = fopen($filepath, 'r');
      if ($handle) {
        $file = File::create([]);
        $file_path = $this->fileSystem->copy($filepath, $this->getUploadPath() . '/' . basename($filepath), FileSystemInterface::EXISTS_RENAME);
        $file->setFilename(basename($filepath));
        $file->setMimeType($mime_type);
        $file->setFileUri($file_path);
        $file->setTemporary();
        $file->save();
        fclose($handle);
        return $file;
      }
    }
    return NULL;
  }

  /**
   * Get file from URL.
   *
   * @param string $fileurl
   *   The file URL.
   * @param array $mime_types
   *   Array of allowed mime types.
   *
   * @return \Drupal\file\FileInterface|null
   *   The file entity or NULL.
   */
  public function getFileFromUrl(string $fileurl, array $mime_types): ?FileInterface {
    $context = stream_context_create(['http' => ['method' => 'HEAD']]);
    $headers = array_change_key_case(get_headers($fileurl, TRUE, $context));
    // Only allowing file types specified in the ViewerType plugin.
    if (!empty($headers['content-type']) && in_array($headers['content-type'], $mime_types)) {
      $lines = file($fileurl);
      if ($lines) {
        $contents = '';
        foreach ($lines as $line) {
          $contents .= $line;
        }
        $fileRepository = \Drupal::service('file.repository');
        $file = $fileRepository->writeData($contents, $this->getUploadPath() . '/' . basename($fileurl), FileSystemInterface::EXISTS_RENAME);
        $file->setTemporary();
        $file->save();
        return $file;
      }
    }
    return NULL;
  }

  /**
   * Tell how to get a file object for further processing.
   *
   * @param mixed $file
   *   The file ID or path.
   * @param array $settings
   *   The settings array.
   * @param mixed $type_plugin
   *   The type plugin.
   * @param string|null $source_type
   *   The source type.
   *
   * @return \Drupal\file\FileInterface|null
   *   The file entity or NULL.
   */
  public function getFile(mixed $file, array $settings, mixed $type_plugin, ?string $source_type = NULL): ?FileInterface {
    return $this->entityTypeManager->getStorage('file')->load($file);
  }

  /**
   * Build batch process items.
   *
   * @return array
   *   The batch items.
   */
  public function buildManualBatchItems(): array {
    $items = [];
    $items[] = ['\Drupal\viewer\Services\Batch::upload', [$this]];
    return $items;
  }

  /**
   * Build metadata for the file.
   *
   * @param \Drupal\file\Entity\File $file
   *   The file entity.
   * @param array $settings
   *   The settings array.
   *
   * @return array
   *   The metadata array.
   */
  public function getMetadata(File $file, array $settings = []): array {
    return [];
  }

  /**
   * Encrypt string.
   *
   * @param string $string
   *   The string to encrypt.
   *
   * @return string
   *   The encrypted string.
   */
  public function encryptString(string $string): string {
    return openssl_encrypt($string, "AES-128-ECB", $this->encryptKey());
  }

  /**
   * Decrypt string.
   *
   * @param string $encrypted_string
   *   The encrypted string.
   *
   * @return string
   *   The decrypted string.
   */
  public function decryptString(string $encrypted_string): string {
    return openssl_decrypt($encrypted_string, "AES-128-ECB", $this->encryptKey());
  }

  /**
   * Encryption/Decryption key.
   *
   * @return string
   *   The encryption key.
   */
  public function encryptKey(): string {
    return Settings::get('hash_salt', $this->getPluginId());
  }

  /**
   * Get upload path based on system settings.
   *
   * @return string
   *   The upload path.
   */
  protected function getUploadPath(): string {
    $path = ViewerInterface::PUBLIC_URI;
    if (\Drupal::service('stream_wrapper_manager')->isValidScheme('private')) {
      $path = ViewerInterface::PRIVATE_URI;
    }
    $this->fileSystem->prepareDirectory($path, FileSystemInterface::CREATE_DIRECTORY);
    return $path;
  }

}
