<?php

namespace Drupal\viewer\Plugin;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\DependencyInjection\DependencySerializationTrait;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\file\Entity\File;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * ViewerTypeBase plugin base class.
 */
class ViewerTypeBase extends PluginBase implements ViewerTypeInterface, ContainerFactoryPluginInterface {

  use DependencySerializationTrait;
  use StringTranslationTrait;

  /**
   * The file system service.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected FileSystemInterface $fileSystem;

  /**
   * Constructs a ViewerTypeBase plugin.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\File\FileSystemInterface $file_system
   *   The file system service.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, FileSystemInterface $file_system) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->fileSystem = $file_system;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static($configuration, $plugin_id, $plugin_definition,
      $container->get('file_system')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getName(): TranslatableMarkup|string|null {
    return $this->pluginDefinition['name'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getExtensions(): array {
    return !empty($this->pluginDefinition['extensions']) ? $this->pluginDefinition['extensions'] : [];
  }

  /**
   * Get the default viewer plugin ID.
   *
   * @return string|null
   *   The default viewer plugin ID.
   */
  public function getDefaultViewerPluginId(): ?string {
    return $this->pluginDefinition['default_viewer'] ?? NULL;
  }

  /**
   * {@inheritdoc}
   */
  public function getExtensionsAsValidator(): array {
    return ['extensions' => implode(' ', array_values($this->getExtensions()))];
  }

  /**
   * Build properties form.
   *
   * @param array $settings
   *   The settings array.
   *
   * @return array
   *   The form array.
   */
  public function propertiesForm(array $settings = []): array {
    return [];
  }

  /**
   * Submit properties form.
   *
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   *
   * @return array
   *   The settings array.
   */
  public function submitPropertiesForm(FormStateInterface $form_state): array {
    return [];
  }

  /**
   * Get metadata from file.
   *
   * @param \Drupal\file\Entity\File $file
   *   The file entity.
   * @param array $settings
   *   The settings array.
   *
   * @return array
   *   The metadata array.
   */
  public function getMetadata(File $file, array $settings = []): array {
    return [];
  }

  /**
   * Get file content as array.
   *
   * @param \Drupal\file\Entity\File $file
   *   The file entity.
   * @param array $settings
   *   The settings array.
   *
   * @return array
   *   The content as array.
   */
  public function getContentAsArray(File $file, array $settings = []): array {
    $array = [];
    $handle = fopen($this->fileSystem->realpath($file->getFileUri()), "r");
    if ($handle !== FALSE) {
      $row = 0;
      while (($data = fgetcsv($handle, 1024, $settings['delimiter'], $settings['enclosure'], $settings['escape'])) !== FALSE) {
        $count = count($data);
        for ($c = 0; $c < $count; $c++) {
          $array[$row][$c] = mb_convert_encoding($data[$c], 'UTF-8', 'UTF-8');
        }
        $row++;
      }
      fclose($handle);
    }
    return $array;
  }

  /**
   * {@inheritdoc}
   */
  public function requirementsMet(): bool {
    return TRUE;
  }

}
