<?php

namespace Drupal\viewer\Plugin\viewer\type;

use Composer\InstalledVersions;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\file\Entity\File;
use Drupal\viewer\Attribute\ViewerType;
use Drupal\viewer\Plugin\ViewerTypeBase;
use PhpOffice\PhpSpreadsheet\IOFactory;

/**
 * Viewer Type plugin.
 *
 * Compatible with PhpSpreadsheet 1.x, 2.x, 3.x, 4.x, and 5.x.
 */
#[ViewerType(
  id: 'xlsx',
  name: new TranslatableMarkup('XLSX'),
  default_viewer: 'spreadsheet_tabs',
)]
class Xlsx extends ViewerTypeBase {

  /**
   * {@inheritdoc}
   */
  public function requirementsMet(): bool {
    return class_exists('PhpOffice\PhpSpreadsheet\Spreadsheet', FALSE);
  }

  /**
   * {@inheritdoc}
   */
  public function getExtensions(): array {
    return [
      'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet' => 'xlsx',
      'application/vnd.ms-excel' => 'xls',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function getMetadata(File $file, $settings = []): array {
    $metadata = [];
    $rows = $this->getContentAsArray($file, $settings);
    foreach (array_keys($rows) as $worksheet) {
      $metadata[$worksheet] = current($rows[$worksheet]);
    }
    return $metadata;
  }

  /**
   * {@inheritdoc}
   */
  public function getContentAsArray(File $file, $settings = []): array {
    $reader = $this->createReader($file);
    $spreadsheet = $reader->load($this->fileSystem->realpath($file->getFileUri()));
    $sheet_columns = [];
    $loadedSheetNames = $spreadsheet->getSheetNames();
    foreach ($loadedSheetNames as $sheet_name) {
      $spreadsheet->setActiveSheetIndexByName($sheet_name);
      $sheetData = $spreadsheet->getActiveSheet()->toArray(null, true, true, true);
      if (!empty($sheetData[1])) {
        $index = 0;
        foreach ($sheetData as $value) {
          if (!empty($value)) {
            foreach ($value as $v) {
              $sheet_columns[$sheet_name][$index][] = $v;
            }
            $index++;
          }
        }
      }
    }
    return $sheet_columns;
  }

  /**
   * Create the appropriate reader for the file.
   *
   * @param \Drupal\file\Entity\File $file
   *   The file entity.
   *
   * @return \PhpOffice\PhpSpreadsheet\Reader\IReader
   *   The reader instance.
   */
  protected function createReader(File $file): \PhpOffice\PhpSpreadsheet\Reader\IReader {
    $readerType = str_contains($file->getFilename(), '.xlsx') ? 'Xlsx' : 'Xls';
    $reader = IOFactory::createReader($readerType);

    // PhpSpreadsheet 5.x requires explicit permission for external images.
    // This maintains backwards compatibility with 1.x-4.x.
    if ($this->getPhpSpreadsheetMajorVersion() >= 5) {
      if (method_exists($reader, 'setAllowExternalImages')) {
        $reader->setAllowExternalImages(false);
      }
    }

    return $reader;
  }

  /**
   * Get the major version of PhpSpreadsheet.
   *
   * @return int
   *   The major version number (1, 2, 3, 4, 5, etc.).
   */
  protected function getPhpSpreadsheetMajorVersion(): int {
    // Try Composer's InstalledVersions first (most reliable).
    if (class_exists(InstalledVersions::class)) {
      try {
        $version = InstalledVersions::getVersion('phpoffice/phpspreadsheet');
        if ($version !== null) {
          return (int) explode('.', $version)[0];
        }
      }
      catch (\Exception $e) {
        // Fall through to default.
      }
    }

    // Default to version 1 for backwards compatibility.
    return 1;
  }

}
