<?php

namespace Drupal\views_adjustable_table\Plugin\views\style;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\style\Table;
use Drupal\views\ViewExecutable;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\Views;

/**
 * Style plugin to render each item as a row in a table with adjustable columns.
 *
 * @ingroup views_style_plugins
 *
 * @ViewsStyle(
 *   id = "adjustable_table",
 *   title = @Translation("Adjustable table"),
 *   help = @Translation("Displays rows in a table with adjustable columns."),
 *   theme = "views_view_table",
 *   display_types = {"normal"}
 * )
 */
class AdjustableTable extends Table {

  public $originalFields = NULL;
  public $originalColumns = NULL;

  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, array &$options = NULL) {
    parent::init($view, $display, $options);

    $this->originalFields = isset($view->field) ? $view->field : [];
    $this->originalColumns = isset($this->options['columns']) ? $this->options['columns'] : [];

    $options = ['id' => 'at_selection'];

    $handler = Views::handlerManager('filter')->createInstance('at_selection', $options);
    $handler->init($this->view, $this->displayHandler, $options);
    $view->display_handler->handlers['adjustable_table']['selection'] = $handler;
    $view->adjustable_table = ['selection' => $handler];
    $input = $view->getExposedInput();
    if (empty($input['columns'])) {
      $input['columns'] = $this->getPreselectedColumns();
      $view->setExposedInput($input);
    }

  }

  /**
   * {@inheritdoc}
   */
  public function defineOptions() {
    $options = parent::defineOptions();
    $options['column_selection_widget'] = ['default' => 'bsmselect'];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    if (empty($form['info'])) {
      return;
    }

    foreach (array_keys($form['info']) as $field) {
      $safe = str_replace(['][', '_', ' '], '-', $field);
      // The $id of the column for dependency checking.
      $id = 'edit-style-options-columns-' . $safe;

      $form['info'][$field]['preselected_column'] = [
        '#type' => 'checkbox',
        '#default_value' => isset($this->options['info'][$field]['preselected_column']) ? $this->options['info'][$field]['preselected_column'] : FALSE,
        '#dependency' => [$id => [$field]],
      ];
    }

    $form['#theme'] = 'views_ui_adjustable_table_style_plugin';

    $form['description_markup']['#markup'] .= '<div class="description form-item">' .
      t('Choose preselected columns which will be shown initially and when the user selects empty set of columns. If no preselected columns are chosen, the first column will be shown.') .
      '</div>';

    $selection_widgets = [
      'bsmselect' => 'bsmSelect',
    ];

    $form['column_selection_widget'] = [
      '#type' => 'select',
      '#title' => t('Column selection widget'),
      '#description' => t('Widget for selecting columns by the end user.'),
      '#options' => $selection_widgets,
      '#default_value' => !empty($this->options['column_selection_widget']) ? $this->options['column_selection_widget'] : 'bsmselect',
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    parent::query();
    $input = $this->view->getExposedInput();
    if (empty($input['columns'])) {
      $input['columns'] = $this->getPreselectedColumns();

      if (empty($input['columns'])) {
        return;
      }
    }

    $options = $this->options;

    if (empty($options['info'])) {
      return;
    }

    $fields = $this->getFieldHandlers();
    if (empty($fields)) {
      $fields = $this->displayHandler->getOption('fields');
    }

    if (empty($fields)) {
      return;
    }

    $columns = $this->sanitizeColumns($options['columns'], $fields);
    $selected_column_fields = [];

    $fields_by_key = [];
    foreach (array_keys($fields) as $no => $field) {
      $fields_by_key['f' . $no] = $field;
    }
    foreach ($input['columns'] as $field_key) {
      if (!array_key_exists($field_key, $fields_by_key)) {
        continue;
      }
      $selected_column_fields[$fields_by_key[$field_key]] = $fields_by_key[$field_key];
    }

    $new_fields = [];
    $new_columns = [];

    foreach ($selected_column_fields as $selected_field) {
      foreach ($this->view->display_handler->handlers['field'] as $field => $field_handler) {
        if ((empty($field_handler->options['exclude'])) && (isset($columns[$field]) && ($columns[$field] == $selected_field))) {
          $new_columns[$field] = $columns[$field];
          if (!array_key_exists($field, $new_fields)) {
            $new_fields[$field] = $field_handler;
          }
        }
      }
    }
    if (empty($new_fields)) {
      return;
    }

    $this->view->display_handler->handlers['field'] = $new_fields;
    $this->options['columns'] = $new_columns;
  }

  /**
   * Helper function for getting preselected columns.
   *
   * @return array
   *   Preselected columns.
   */
  private function getPreselectedColumns() {
    $options = $this->options;

    if (empty($options['info'])) {
      return [];
    }

    $fields = $this->getFieldHandlers();
    $columns = $this->sanitizeColumns($options['columns'], $fields);
    $field_nos = [];
    $ids = [];
    $preselected = [];

    foreach (array_keys($fields) as $field_no => $field) {
      $field_nos[$field] = $field_no;
    }

    foreach ($columns as $field => $column) {
      if ($field == $column && empty($fields[$field]->options['exclude'])) {
        $id = 'f' . $field_nos[$field];
        $ids[] = $id;
        if (!empty($options['info'][$field]['preselected_column'])) {
          $preselected[] = $id;
        }
      }
    }

    if (empty($preselected) && !empty($ids)) {
      reset($ids);
      $preselected = [current($ids)];
    }

    return $preselected;
  }

  /**
   * Helper function for getting view's field handlers.
   *
   * @return array
   *   Preselected columns.
   */
  private function getFieldHandlers() {
    $display_id = $this->view->current_display;
    return !empty($this->view->display_handler->handlers['field']) ? $this->view->display_handler->handlers['field'] : $this->view->getHandlers('field', $display_id);
  }

}
