<?php

declare(strict_types=1);

namespace Drupal\views_area_library\Plugin\views\area;

use Drupal\Core\Asset\LibraryDiscoveryInterface;
use Drupal\Core\Extension\ThemeHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\views\Attribute\ViewsArea;
use Drupal\views\Plugin\views\area\AreaPluginBase;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Views area library handler.
 *
 * @ingroup views_area_handlers
 */
#[ViewsArea('library')]
class Library extends AreaPluginBase {

  /**
   * The library discovery.
   */
  protected LibraryDiscoveryInterface $libraryDiscovery;

  /**
   * The theme handler.
   */
  protected ThemeHandlerInterface $themeHandler;

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    $instance = parent::create($container, $configuration, $plugin_id, $plugin_definition);
    $instance->libraryDiscovery = $container->get('library.discovery');
    $instance->themeHandler = $container->get('theme_handler');
    return $instance;
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions(): array {
    $options = parent::defineOptions();
    $options['library'] = ['default' => ''];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function adminSummary() {
    if (!$library = $this->options['library']) {
      return $this->t('No library selected');
    }
    $parts = explode('/', $library, 2);
    if (\count($parts) !== 2) {
      $library .= " [{$this->t('Invalid format')}]";
    }
    elseif (!$this->libraryDiscovery->getLibraryByName($parts[0], $parts[1])) {
      $library .= " [{$this->t('Library not found')}]";
    }
    return $library;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    $form['library'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Library'),
      '#default_value' => $this->options['library'],
      '#required' => TRUE,
      '#description' => $this->t('Library must be in the format "<code>extension/library</code>". For example "<code>core/drupal.dialog.ajax</code>".'),
      '#placeholder' => 'core/drupal.dialog.ajax',
    ];
    $form['helpful_links'] = [
      '#theme' => 'links',
      '#heading' => [
        'text' => $this->t('Helpful links'),
        'level' => 'strong',
      ],
      '#links' => [
        [
          'title' => $this->t('Adding assets (CSS, JS) to a Drupal module via *.libraries.yml'),
          'url' => Url::fromUri('https://www.drupal.org/node/2274843'),
          'attributes' => ['target' => '_blank'],
        ],
        [
          'title' => $this->t('Adding assets (CSS, JS) to a Drupal theme via *.libraries.yml'),
          'url' => Url::fromUri('https://www.drupal.org/node/2216195'),
          'attributes' => ['target' => '_blank'],
        ],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(&$form, FormStateInterface $form_state): void {
    $library = $form_state->getValue(['options', 'library']);
    $parts = explode('/', $library, 2);
    if (\count($parts) !== 2) {
      $form_state->setErrorByName('library', $this->t('Library must be in the format "<code>extension/library</code>".'));
      return;
    }
    if (!$this->libraryDiscovery->getLibraryByName($parts[0], $parts[1])) {
      $form_state->setErrorByName('library', $this->t('The library "<code>@library</code>" does not exist.', ['@library' => $library]));
    }
  }

  /**
   * {@inheritdoc}
   */
  public function render($empty = FALSE): array {
    $build = [];
    if ($library = $this->options['library']) {
      $build['#attached']['library'][] = $library;
    }
    return $build;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies(): array {
    $dependencies = parent::calculateDependencies();
    $dependency = explode('/', $this->options['library'])[0];
    if ($this->getModuleHandler()->moduleExists($dependency)) {
      $dependencies['module'][] = $dependency;
    }
    elseif ($this->themeHandler->themeExists($dependency)) {
      $dependencies['theme'][] = $dependency;
    }
    return $dependencies;
  }

}
