<?php

namespace Drupal\views_argument_between_daterange\Plugin\views\argument;

use Drupal\views\Plugin\views\argument\FullDate;
use Drupal\Core\Datetime\DrupalDateTime;

/**
 * Between fulldate views contextual filter.
 *
 * Contextual filter: treat a CCYYMMDD argument as a date and return nodes
 *  where that date lies between the start and end of a daterange field.
 *
 * @ViewsArgument("date_range_full_date_between_argument")
 */
class DateRangeFullDateBetweenArgument extends FullDate {

  /**
   * {@inheritdoc}
   */
  public function query($group_by = FALSE) {

    $this->ensureMyTable();

    $date = $this->getDateArgument();
    if (!$date) {
      return;
    }

    $field     = $this->realField;
    $start_col = "{$this->tableAlias}.{$field}_value";
    $end_col   = "{$this->tableAlias}.{$field}_end_value";

    // Format the argument as Ymd (same as FullDate)
    $contextual_day = $date->format('Ymd');

    // Use getDateField() to handle site timezone offset automatically.
    $start_expr = $this->query->getDateField($start_col, TRUE, TRUE);
    $end_expr   = $this->query->getDateField($end_col, TRUE, TRUE);

    // Build WHERE clause.
    $where = "(
        ({$end_expr} IS NULL AND DATE_FORMAT({$start_expr}, '%Y%m%d') = :contextual_day)
        OR
        ({$end_expr} IS NOT NULL AND DATE_FORMAT({$start_expr}, '%Y%m%d') <= :contextual_day
                                 AND DATE_FORMAT({$end_expr}, '%Y%m%d') >= :contextual_day)
    )";

    $group = $this->options['group'] ?? 0;

    $this->query->addWhereExpression($group, $where, [
      ':contextual_day' => $contextual_day,
    ]);
  }

  /**
   * Get the contextual date as a DrupalDateTime.
   *
   * @return \Drupal\Core\Datetime\DrupalDateTime|null
   *   The contextual date or NULL.
   */
  protected function getDateArgument() {
    if (!empty($this->argument)) {
      try {
        return new DrupalDateTime($this->argument);
      }
      catch (\Exception $e) {
        return NULL;
      }
    }
    return NULL;
  }

}
