<?php

namespace Drupal\views_base_url\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\path_alias\AliasManagerInterface;
use Drupal\views\Attribute\ViewsField;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\Core\Url;
use Drupal\Core\Link;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * A handler to generate custom internal URLs.
 *
 * @ingroup views_field_handlers
 */
#[ViewsField("custom_internal_url")]
class CustomInternalUrl extends FieldPluginBase {

  use TokenTrait;

  /**
   * The route for the front page.
   *
   * This is used when no link path is provided.
   *
   * @var string
   */
  protected const ROUTE_FRONT = 'route:<front>';

  /**
   * {@inheritdoc}
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected AliasManagerInterface $pathAliasManager,
    protected LanguageManagerInterface $languageManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('path_alias.manager'),
      $container->get('language_manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function query() {}

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();

    $options['url_path'] = ['default' => ''];
    $options['url_query'] = ['default' => ''];
    $options['url_fragment'] = ['default' => ''];
    $options['render_as_link'] = ['default' => FALSE];
    $options['link_options']['contains'] = [
      'link_text' => ['default' => ''],
      'link_class' => ['default' => ''],
      'link_title' => ['default' => ''],
      'link_rel' => ['default' => ''],
      'link_target' => ['default' => ''],
    ];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    $form['url_path'] = [
      '#type' => 'textfield',
      '#title' => $this->t('URL path'),
      '#description' => $this->t('Drupal path to use for the URL (e.g., <code>@example1</code>).<br>If nothing is provided, <code>&lt;front&gt;</code> will be used.', ['@example1' => '/node/{{ nid }}']),
      '#default_value' => $this->options['url_path'],
    ];

    $form['url_query'] = [
      '#type' => 'textfield',
      '#title' => $this->t('URL query parameters'),
      '#description' => $this->t('Attach queries to the link. If there are multiple queries separate them using a space or &amp;. For eg: %example1 OR %example2', [
        '%example1' => 'destination=node/add/page',
        '%example2' => 'destination=node/add/page q=some/page',
      ]),
      '#default_value' => $this->options['url_query'],
    ];

    $form['url_fragment'] = [
      '#type' => 'textfield',
      '#title' => $this->t('URL Fragment'),
      '#description' => $this->t('Provide the ID with which you want to create fragment link.'),
      '#default_value' => $this->options['url_fragment'],
    ];

    $form['render_as_link'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Render as link'),
      '#description' => $this->t('Renders the generated URL as an HTML link.'),
      '#default_value' => $this->options['render_as_link'],
    ];

    $form['link_options'] = [
      '#type' => 'container',
      '#states' => [
        'invisible' => [
          ':input[type=checkbox][name="options[render_as_link]"]' => ['checked' => FALSE],
        ],
      ],
    ];

    $form['link_options']['link_text'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Link text'),
      '#description' => $this->t('Link text. If nothing provided then link path will appear as link text.'),
      '#default_value' => $this->options['link_options']['link_text'],
    ];

    $form['link_options']['link_class'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Link class'),
      '#description' => $this->t('CSS class to be applied to this link.'),
      '#default_value' => $this->options['link_options']['link_class'],
    ];

    $form['link_options']['link_title'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Link title'),
      '#description' => $this->t('Title attribute for this link.'),
      '#default_value' => $this->options['link_options']['link_title'],
    ];

    $form['link_options']['link_rel'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Link rel'),
      '#description' => $this->t('Rel attribute for this link.'),
      '#default_value' => $this->options['link_options']['link_rel'],
    ];

    $form['link_options']['link_target'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Link target'),
      '#description' => $this->t('Target attribute for this link.'),
      '#default_value' => $this->options['link_options']['link_target'],
    ];

    // This construct uses 'hidden' and not markup because process doesn't
    // run. It also has an extra div because the dependency wants to hide
    // the parent in situations like this, so we need a second div to
    // make this work.
    $form['link_options']['help'] = [
      '#type' => 'details',
      '#title' => $this->t('Replacement patterns'),
      '#value' => $this->getReplacementTokens(),
    ];

    parent::buildOptionsForm($form, $form_state);
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $language = $this->languageManager->getCurrentLanguage();

    $tokens = $this->getRenderTokens('');

    // Link path.
    if (!empty($this->options['url_path'])) {
      $aliased_path = $this->simpleTokenReplace($this->options['url_path'], $tokens);
      $aliased_path = $this->pathAliasManager->getAliasByPath($aliased_path);
      $link_uri = 'internal:' . $aliased_path;
    }
    else {
      $link_uri = self::ROUTE_FRONT;
    }

    // Build the URL.
    $url = Url::fromUri($link_uri, [
      'language' => $language,
      'absolute' => TRUE,
    ]);

    // Link query.
    if (!empty($this->options['url_query'])) {
      $url_query = trim($this->simpleTokenReplace($this->options['url_query'], $tokens));
      $query_items = preg_split('/[ &]+/', $url_query, -1, PREG_SPLIT_NO_EMPTY);
      $query = [];
      foreach ($query_items as $query_item) {
        $param = explode('=', $query_item);
        $query[$param[0]] = $param[1];
      }
      if (count($query) > 0) {
        $url->setOption('query', $query);
      }
    }

    // Link fragment.
    if (!empty($this->options['url_fragment'])) {
      $url_fragment = trim($this->simpleTokenReplace($this->options['url_fragment'], $tokens));
      if (!empty($url_fragment)) {
        $url->setOption('fragment', $url_fragment);
      }
    }

    if ($this->options['render_as_link']) {

      // Link attributes.
      $attributes = [];
      if (!empty($this->options['link_options']['link_class'])) {
        $link_class = trim($this->simpleTokenReplace($this->options['link_options']['link_class'], $tokens));
        if (!empty($link_class)) {
          $attributes['class'] = explode(' ', $link_class);
        }
      }
      if (!empty($this->options['link_options']['link_title'])) {
        $link_title = trim($this->simpleTokenReplace($this->options['link_options']['link_title'], $tokens));
        if (!empty($link_title)) {
          $attributes['title'] = $link_title;
        }
      }
      if (!empty($this->options['link_options']['link_rel'])) {
        $attributes['rel'] = $this->options['link_options']['link_rel'];
      }
      if (!empty($this->options['link_options']['link_target'])) {
        $attributes['target'] = $this->options['link_options']['link_target'];
      }

      $url->setOption('attributes', $attributes);

      // Link text.
      if (!empty($this->options['link_options']['link_text'])) {
        $link_text = [
          '#markup' => $this->viewsTokenReplace($this->options['link_options']['link_text'], $tokens),
        ];
      }
      else {
        $link_text = [
          '#plain_text' => strtok($url->toString(), '?'),
        ];
      }

      // Generate the HTML link.
      return [
        '#markup' => Link::fromTextAndUrl($link_text, $url)->toString(),
      ];
    }
    else {
      return [
        '#plain_text' => $url->toString(),
      ];
    }
  }

}
