<?php

namespace Drupal\Tests\views_base_url\Functional;

use Drupal\Component\Utility\Random;
use Drupal\Core\Url;
use Drupal\Tests\BrowserTestBase;
use Drupal\Tests\TestFileCreationTrait;

/**
 * Basic test for the custom internal url views field.
 *
 * @group views_base_url
 */
class CustomInternalUrlFieldTest extends BrowserTestBase {

  use TestFileCreationTrait {
    getTestFiles as drupalGetTestFiles;
  }

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * A user with various administrative privileges.
   *
   * @var \Drupal\user\UserInterface
   */
  protected $adminUser;

  /**
   * The installation profile to use with this test.
   *
   * This test class requires the "tags" taxonomy field.
   *
   * @var string
   */
  protected $profile = 'standard';

  /**
   * Node count.
   *
   * Number of nodes to be created in the tests.
   *
   * @var int
   */
  protected $nodeCount = 3;

  /**
   * Nodes.
   *
   * The nodes that is going to be created in the tests.
   *
   * @var array
   */
  protected $nodes;

  /**
   * Path alias storage.
   *
   * @var \Drupal\path_alias\PathAliasStorage
   */
  protected $pathAliasStorage;

  /**
   * Path alias manager.
   *
   * @var \Drupal\path_alias\AliasManager
   */
  protected $pathAliasManager;

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'custom_internal_url_test',
  ];

  /**
   * Definition of File System Interface.
   *
   * @var \Drupal\Core\File\FileSystemInterface
   */
  protected $fileSystem;

  /**
   * {@inheritdoc}
   */
  public function setUp(): void {
    parent::setUp();

    $this->adminUser = $this->drupalCreateUser([
      'create article content',
    ]);
    $random = new Random();

    /** @var \Drupal\path_alias\PathAliasStorage $pathAliasStorage */
    $this->pathAliasStorage = $this->container->get('entity_type.manager')->getStorage('path_alias');
    /** @var \Drupal\path_alias\AliasManager $pathAliasManager */
    $this->pathAliasManager = $this->container->get('path_alias.manager');
    /** @var \Drupal\Core\File\FileSystemInterface $fileSystem */
    $this->fileSystem = $this->container->get('file_system');
    // Create $this->nodeCount nodes.
    $this->drupalLogin($this->adminUser);
    for ($i = 1; $i <= $this->nodeCount; $i++) {
      // Create node.
      $title = $random->name();
      $image = current($this->drupalGetTestFiles('image'));
      $edit = [
        'title[0][value]' => $title,
        'files[field_image_0]' => $this->fileSystem->realpath($image->uri),
      ];
      $this->drupalGet('node/add/article');
      $this->submitForm($edit, t('Save'));
      $this->submitForm(['field_image[0][alt]' => $title], t('Save'));

      $this->nodes[$i] = $this->drupalGetNodeByTitle($title);
      $path_alias = $this->pathAliasStorage->create([
        'path' => '/node/' . $this->nodes[$i]->id(),
        'alias' => '/content/' . $title,
      ]);
      $path_alias->save();
    }
    $this->drupalLogout();
  }

  /**
   * Returns the base URL of the site.
   */
  protected function getBaseUrl() {
    $url = Url::fromRoute(
      '<front>',
      [],
      [
        'absolute' => TRUE,
      ]
    );
    return $url->toString();
  }

  /**
   * Tests views base url field when `show_link` enabled and no link settings.
   */
  protected function assertCustomInternalUrlLinkNoSettings() {
    $this->drupalGet('custom-internal-url-link-no-settings-test');
    $this->assertSession()->statusCodeEquals(200);

    $elements = $this->xpath('//div[contains(@class,"views-element-container")]/div/div[contains(@class,"views-row")]');
    $this->assertEquals($this->nodeCount, count($elements), t('There are @count rows', [
      '@count' => $this->nodeCount,
    ]));

    // The link generated by the Url class contains a trailing slash.
    $link_path = $this->getBaseUrl();
    $link_text = $link_path;
    $elements = $this->xpath('//a[@href=:path and text()=:text]', [
      ':path' => $link_path,
      ':text' => $link_text,
    ]);
    $this->assertEquals($this->nodeCount, count($elements), 'Views base url rendered as link with no settings set');
  }

  /**
   * Tests views base url field when `show_link` is disabled.
   */
  public function testCustomInternalUrlNoLink() {
    $this->drupalGet('custom-internal-url-nolink-test');
    $this->assertSession()->statusCodeEquals(200);

    $elements = $this->xpath('//div[contains(@class,"views-element-container")]/div/div[contains(@class,"views-row")]');
    $this->assertEquals($this->nodeCount, count($elements), t('There are @count rows', [
      '@count' => $this->nodeCount,
    ]));

    foreach ($this->nodes as $node) {
      $link_path = Url::fromUri(
        'internal:' . $this->pathAliasManager->getAliasByPath('/node/' . $node->id()),
        [
          'fragment' => 'new',
          'query' => [
            'destination' => 'node',
          ],
          'absolute' => TRUE,
        ]
      )->toString();

      $elements = $this->xpath('//div[contains(@class,"views-field-custom-internal-url")]/span[@class="field-content" and text()=:value]', [
        ':value' => $link_path,
      ]);
      $this->assertEquals(1, count($elements), t('Custom internal url displayed as text with all settings'));
    }
  }

  /**
   * Tests views base url field when `show_link` enabled and all settings set.
   */
  public function testCustomInternalUrlLinkAllSettings() {
    $this->drupalGet('custom-internal-url-link-all-settings-test');
    $this->assertSession()->statusCodeEquals(200);

    $elements = $this->xpath('//div[contains(@class,"views-element-container")]/div/div[contains(@class,"views-row")]');
    $this->assertEquals($this->nodeCount, count($elements), t('There are @count rows', [
      '@count' => $this->nodeCount,
    ]));

    foreach ($this->nodes as $node) {
      $link_class = 'custom-internal-url-test';
      $link_title = $node->getTitle();
      $link_text = $node->getTitle();
      $link_rel = 'rel-attribute';
      $link_target = '_blank';
      $link_path = Url::fromUri(
        'internal:' . $this->pathAliasManager->getAliasByPath('/node/' . $node->id()),
        [
          'attributes' => [
            'class' => $link_class,
            'title' => $link_title,
            'rel' => $link_rel,
            'target' => $link_target,
          ],
          'fragment' => 'new',
          'query' => [
            'destination' => 'node',
          ],
          'absolute' => TRUE,
        ]
      )->toString();

      $elements = $this->xpath('//a[@href=:path and @class=:class and @title=:title and @rel=:rel and @target=:target and text()=:text]', [
        ':path' => $link_path,
        ':class' => $link_class,
        ':title' => $link_title,
        ':rel' => $link_rel,
        ':target' => $link_target,
        ':text' => $link_text,
      ]);
      $this->assertEquals(1, count($elements), 'Views base url rendered as link with all settings');
    }
  }

  /**
   * Tests views base url field when `show_link` enabled and `link_path` set.
   */
  public function testCustomInternalUrlLinkLinkPath() {
    $this->drupalGet('custom-internal-url-link-link-path-test');
    $this->assertSession()->statusCodeEquals(200);

    $elements = $this->xpath('//div[contains(@class,"views-element-container")]/div/div[contains(@class,"views-row")]');
    $this->assertEquals($this->nodeCount, count($elements), t('There are @count rows', [
      '@count' => $this->nodeCount,
    ]));

    foreach ($this->nodes as $node) {
      $link_path = Url::fromUri(
        'internal:' . $this->pathAliasManager->getAliasByPath('/node/' . $node->id()),
        [
          'absolute' => TRUE,
        ]
      )->toString();
      $link_text = $link_path;

      $elements = $this->xpath('//a[@href=:path and text()=:text]', [
        ':path' => $link_path,
        ':text' => $link_text,
      ]);
      $this->assertEquals(1, count($elements), 'Views base url rendered as link with link path set');
    }
  }

  /**
   * Tests views base url field when `show_link` enabled and no `link_path`.
   */
  public function testCustomInternalUrlLinkNoLinkPath() {
    $this->assertCustomInternalUrlLinkNoSettings();
  }

  /**
   * Tests views base url field when `show_link` enabled and `link_text` set.
   */
  public function testCustomInternalUrlLinkLinkText() {
    $this->drupalGet('custom-internal-url-link-link-text-test');
    $this->assertSession()->statusCodeEquals(200);

    $elements = $this->xpath('//div[contains(@class,"views-element-container")]/div/div[contains(@class,"views-row")]');
    $this->assertEquals($this->nodeCount, count($elements), t('There are @count rows', [
      '@count' => $this->nodeCount,
    ]));

    foreach ($this->nodes as $node) {
      $link_path = $this->getBaseUrl();
      $link_text = $node->getTitle();

      $elements = $this->xpath('//a[@href=:path and text()=:text]', [
        ':path' => $link_path,
        ':text' => $link_text,
      ]);
      $this->assertEquals(1, count($elements), 'Views base url rendered as link with link text set');
    }
  }

  /**
   * Tests views base url field when `show_link` enabled and no `link_text` set.
   */
  public function testCustomInternalUrlLinkNoLinkText() {
    $this->assertCustomInternalUrlLinkNoSettings();
  }

  /**
   * Tests views base url field when `show_link` enabled and `link_query` set.
   */
  public function testCustomInternalUrlLinkLinkQuery() {
    $this->drupalGet('custom-internal-url-link-link-query-test');
    $this->assertSession()->statusCodeEquals(200);

    $elements = $this->xpath('//div[contains(@class,"views-element-container")]/div/div[contains(@class,"views-row")]');
    $this->assertEquals($this->nodeCount, count($elements), t('There are @count rows', [
      '@count' => $this->nodeCount,
    ]));

    $link_path = $this->getBaseUrl();
    $link_text = $link_path;

    $elements = $this->xpath('//a[@href=:path and text()=:text]', [
      ':path' => Url::fromUri(
        $link_path,
        [
          'query' => [
            'destination' => 'node',
          ],
          'absolute' => TRUE,
        ]
      )->toString(),
      ':text' => $link_text,
    ]);
    $this->assertEquals($this->nodeCount, count($elements), 'Views base url rendered as link with link query set');
  }

  /**
   * Tests views base url field when `show_link` enabled and no `link_query`.
   */
  public function testCustomInternalUrlLinkNoLinkQuery() {
    $this->assertCustomInternalUrlLinkNoSettings();
  }

  /**
   * Tests views base url field when rendered as image.
   */
  public function testCustomInternalUrlImage() {
    $this->drupalGet('custom-internal-url-image-test');
    $this->assertSession()->statusCodeEquals(200);

    $elements = $this->xpath('//div[contains(@class,"views-element-container")]/div/div[contains(@class,"views-row")]');
    $this->assertEquals($this->nodeCount, count($elements), t('There are @count rows', [
      '@count' => $this->nodeCount,
    ]));

    foreach ($this->nodes as $node) {
      $field = $node->get('field_image');
      $value = $field->getValue();

      $image_uri = \Drupal::service('file_url_generator')->generateString($field->entity->getFileUri());
      $image_alt = $value[0]['alt'];
      $image_width = $value[0]['width'];
      $image_height = $value[0]['height'];

      $link_class = 'custom-internal-url-test';
      $link_title = $node->getTitle();
      $link_rel = 'rel-attribute';
      $link_target = '_blank';
      $link_path = Url::fromUri(
        'internal:' . $this->pathAliasManager->getAliasByPath('/node/' . $node->id()),
        [
          'attributes' => [
            'class' => $link_class,
            'title' => $link_title,
            'rel' => $link_rel,
            'target' => $link_target,
          ],
          'fragment' => 'new',
          'query' => [
            'destination' => 'node',
          ],
          'absolute' => TRUE,
        ]
      )->toString();

      $elements = $this->xpath('//a[@href=:path and @class=:class and @title=:title and @rel=:rel and @target=:target]/img[@src=:url and @width=:width and @height=:height and @alt=:alt]', [
        ':path' => $link_path,
        ':class' => $link_class,
        ':title' => $link_title,
        ':rel' => $link_rel,
        ':target' => $link_target,
        ':url' => $image_uri,
        ':width' => $image_width,
        ':height' => $image_height,
        ':alt' => $image_alt,
      ]);
      $this->assertEquals(1, count($elements), 'Views base url rendered as link image');
    }
  }

}
