<?php

namespace Drupal\views_conditional_text\Plugin\views\display_extender;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\Plugin\views\display_extender\DisplayExtenderPluginBase;
use Drupal\views\ViewExecutable;
use Drupal\views\Plugin\views\display\Page;
use Drupal\views\Plugin\views\display\Block;

/**
 * Adds a "Query-only (conditional text)" mode to Views displays.
 *
 * @ViewsDisplayExtender(
 *   id = "query_only_mode",
 *   title = @Translation("Query-only (conditional text)"),
 *   help = @Translation("Run the view query without rendering fields, rows, exposed filters, pager, or path."),
 *   no_ui = FALSE
 * )
 */
class QueryOnlyMode extends DisplayExtenderPluginBase {

  /**
   * Whether this extender applies to the current display.
   *
   * @var bool
   */
  protected bool $isApplicable = FALSE;

  /**
   * Displays this extender applies to.
   */
  protected const ACCEPTABLE_DISPLAY_PLUGINS = [
    Page::class,
    Block::class,
  ];

  /**
   * {@inheritdoc}
   */
  public function init(ViewExecutable $view, DisplayPluginBase $display, ?array &$options = NULL) {
    parent::init($view, $display, $options);

    foreach (self::ACCEPTABLE_DISPLAY_PLUGINS as $displayPlugin) {
      if ($display instanceof $displayPlugin) {
        $this->isApplicable = TRUE;
        break;
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function defineOptionsAlter(&$options) {
    if (!$this->isApplicable) {
      return;
    }

    $options['query_only_mode'] = [
      'contains' => [
        'enabled' => ['default' => FALSE],
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function optionsSummary(&$categories, &$options) {
    if (!$this->isApplicable) {
      return;
    }

    $categories['query_only_mode'] = [
      'title' => $this->t('Query-only mode'),
      'column' => 'second',
    ];

    $enabled = !empty($this->options['query_only_mode']['enabled']);
    if ($enabled) {
      // Hide exposed form summary.
      unset($options['exposed_form']);
      unset($options['exposed_block']);
    }

    $options['query_only_mode'] = [
      'category' => 'query_only_mode',
      'title' => $this->t('Query-only mode'),
      'value' => $enabled ? $this->t('Enabled') : $this->t('Disabled'),
      'desc' => $this->t('Runs the query but suppresses fields, rows, exposed filters, pager, and path.'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    if (!$this->isApplicable || $form_state->get('section') !== 'query_only_mode') {
      return;
    }

    $form['#title'] .= $this->t('Query-only mode');

    $form['query_only_mode'] = [
      '#type' => 'container',
      '#tree' => TRUE,
    ];

    $form['query_only_mode']['enabled'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable query-only (conditional text) mode'),
      '#default_value' => $this->options['query_only_mode']['enabled'] ?? FALSE,
      '#description' => $this->t(
        'The view query will run, but fields, rows, exposed filters, pagers, and paths are suppressed. Use Header, Footer, or Empty areas to output conditional text.'
      ),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state) {
    if (!$this->isApplicable || $form_state->get('section') !== 'query_only_mode') {
      return;
    }

    $this->options['query_only_mode'] = $form_state->getValue('query_only_mode');
  }

  /**
   * {@inheritdoc}
   */
  public function preExecute() {
    if (!$this->isApplicable || empty($this->options['query_only_mode']['enabled'])) {
      return;
    }

    $this->displayHandler->setOption('pager', [
      'type' => 'none',
      'options' => [],
    ]);
    $this->displayHandler->setOption('use_more', FALSE);
  }

}
