<?php

declare(strict_types=1);

namespace Drupal\views_conditional_text\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\views\ViewExecutable;

class ViewsConditionalTextHooks {

  /**
   * hook_views_post_render()
   * 
   * Removes rows after rendering so header/footer/empty remain intact.
   */
  #[Hook('views_post_render')]
  public function viewsPostRender(ViewExecutable $view, array &$output): void {
    if (!$this->isQueryOnly($view)) {
      return;
    }

    if (isset($output['#rows'])) {
      $output['#rows'] = [];
    }

    if (isset($output['#children'])) {
      $output['#children'] = '';
    }
  }

  /**
   * Checks whether Query-only mode is enabled for the current display.
   *
   * This is the single source of truth used across all hooks.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The Views executable.
   *
   * @return bool
   *   TRUE if query-only mode is enabled for this display.
   */
  public function isQueryOnly(ViewExecutable $view): bool {
    if (empty($view->display_handler)) {
      return FALSE;
    }

    $extenders = $view->display_handler->getExtenders();

    return !empty($extenders['query_only_mode'])
      && !empty($extenders['query_only_mode']->options['query_only_mode']['enabled']);
  }

}
