<?php

declare(strict_types=1);

namespace Drupal\views_config_field\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\views\Attribute\ViewsField;
use Drupal\Core\Config\ConfigFactoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Cache\CacheableDependencyTrait;
use Drupal\Core\Cache\CacheableDependencyInterface;

/**
 * Provides a field that displays a Drupal config value.
 */
#[ViewsField("views_config_field")]
class ViewsConfigField extends FieldPluginBase implements CacheableDependencyInterface {
  use CacheableDependencyTrait;

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   * */
  protected $configFactory;

  /**
   * Constructor.
   *
   * @param array $configuration
   *   The configuration.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin definition.
   * @param \Drupal\Core\Config\ConfigFactoryInterface $config_factory
   *   The config factory.
   */
  final public function __construct(array $configuration, $plugin_id, $plugin_definition, ConfigFactoryInterface $config_factory) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->configFactory = $config_factory;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('config.factory'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    // This field does not add anything to the query.
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['config_name'] = ['default' => 'system.site'];
    $options['config_key'] = ['default' => 'name'];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $form['config_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Configuration name'),
      '#description' => $this->t('The configuration object to load, e.g. "system.site" or "my_module.settings".'),
      '#default_value' => $this->options['config_name'],
      '#required' => TRUE,
    ];

    $form['config_key'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Configuration key'),
      '#description' => $this->t('The property or nested key to display, e.g. "name" or "email". Use dot notation for nested keys, e.g. "foo.bar.baz".'),
      '#default_value' => $this->options['config_key'],
      '#required' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function render(ResultRow $values) {
    $config_name = $this->options['config_name'];
    $config_key = $this->options['config_key'];

    if (!$config_name || !$config_key) {
      return '';
    }

    $config = $this->configFactory->get($config_name);

    $value = $this->getNestedValue($config->getRawData(), explode('.', $config_key));

    return is_scalar($value) ? $value : json_encode($value);
  }

  /**
   * Helper: Get nested array value using dot notation.
   */
  protected function getNestedValue(array $array, array $keys) {
    foreach ($keys as $key) {
      if (!is_array($array) || !array_key_exists($key, $array)) {
        return NULL;
      }
      $array = $array[$key];
    }
    return $array;
  }

}
