<?php

declare(strict_types=1);

namespace Drupal\Tests\views_config_field\Unit\Plugin\views\field;

use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\ViewExecutable;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\ImmutableConfig;
use Drupal\Core\DependencyInjection\ContainerBuilder;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\views_config_field\Plugin\views\field\ViewsConfigField;

/**
 * Unit tests for the ViewsConfigField plugin.
 *
 * @coversDefaultClass \Drupal\views_config_field\Plugin\views\field\ViewsConfigField
 * @group views_config_field
 */
class ViewsConfigFieldTest extends UnitTestCase {

  /**
   * The config factory mock.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $configFactory;

  /**
   * The config mock.
   *
   * @var \Drupal\Core\Config\ImmutableConfig|\PHPUnit\Framework\MockObject\MockObject
   */
  protected $config;

  /**
   * The views config field plugin.
   *
   * @var \Drupal\views_config_field\Plugin\views\field\ViewsConfigField
   */
  protected $plugin;

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    $container = new ContainerBuilder();

    $this->configFactory = $this->createMock(ConfigFactoryInterface::class);
    $this->config = $this->createMock(ImmutableConfig::class);

    // Mock views.settings config for buildOptionsForm.
    $views_settings = $this->createMock(ImmutableConfig::class);
    $views_settings->method('get')
      ->with('field_rewrite_elements')
      ->willReturn([]);

    // Configure the config factory to return appropriate configs.
    $this->configFactory->method('get')
      ->willReturnCallback(function ($config_name) use ($views_settings) {
        if ($config_name === 'views.settings') {
          return $views_settings;
        }
        return $this->config;
      });

    $container->set('config.factory', $this->configFactory);

    // Add string translation service for buildOptionsForm test.
    $string_translation = $this->getStringTranslationStub();
    $container->set('string_translation', $string_translation);

    // Add module handler service for buildOptionsForm test.
    $module_handler = $this->createMock(ModuleHandlerInterface::class);
    $container->set('module_handler', $module_handler);

    \Drupal::setContainer($container);

    $configuration = [
      'config_name' => 'system.site',
      'config_key' => 'name',
    ];
    $plugin_id = 'views_config_field';
    $plugin_definition = [
      'id' => 'views_config_field',
      'provider' => 'views_config_field',
      'title' => 'Config Field',
      'group' => 'Configuration',
    ];

    $this->plugin = new ViewsConfigField(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $this->configFactory
    );

    // Initialize the plugin options with default values.
    $this->plugin->options = [
      'id' => 'views_config_field',
      'config_name' => 'system.site',
      'config_key' => 'name',
      'admin_label' => '',
      'element_label_colon' => TRUE,
      'exclude' => FALSE,
      'element_type' => '',
      'element_class' => '',
      'element_label_type' => '',
      'element_label_class' => '',
      'element_wrapper_type' => '',
      'element_wrapper_class' => '',
      'element_default_classes' => TRUE,
      'alter' => [
        'alter_text' => FALSE,
        'text' => '',
        'make_link' => FALSE,
        'path' => '',
        'absolute' => FALSE,
        'external' => FALSE,
        'replace_spaces' => FALSE,
        'path_case' => 'none',
        'trim_whitespace' => FALSE,
        'alt' => '',
        'rel' => '',
        'link_class' => '',
        'prefix' => '',
        'suffix' => '',
        'target' => '',
        'nl2br' => FALSE,
        'max_length' => '',
        'word_boundary' => TRUE,
        'ellipsis' => TRUE,
        'more_link' => FALSE,
        'more_link_text' => '',
        'more_link_path' => '',
        'strip_tags' => FALSE,
        'trim' => FALSE,
        'preserve_tags' => '',
        'html' => FALSE,
      ],
      'empty' => '',
      'empty_zero' => FALSE,
      'hide_empty' => FALSE,
      'hide_alter_empty' => TRUE,
    ];

    // Mock the view and display for buildOptionsForm test.
    $view = $this->getMockBuilder(ViewExecutable::class)
      ->disableOriginalConstructor()
      ->getMock();

    $display = $this->getMockBuilder(DisplayPluginBase::class)
      ->disableOriginalConstructor()
      ->getMock();

    // Mock the getHandler method to return null (no other handlers).
    $display->method('getHandler')
      ->willReturn(NULL);

    // Mock the getFieldLabels method to return an empty array.
    $display->method('getFieldLabels')
      ->willReturn([]);

    // Mock additional methods that might be called during buildOptionsForm.
    $display->method('getHandlers')
      ->willReturn([]);

    $display->method('getOption')
      ->willReturn([]);

    // Mock the view's display_handler property.
    $view->display_handler = $display;

    // Mock additional view properties that might be accessed.
    $view_storage = $this->createMock('Drupal\views\Entity\View');
    $view_storage->method('get')
      ->willReturn([]);
    $view->storage = $view_storage;

    $this->plugin->view = $view;
    $this->plugin->displayHandler = $display;
  }

  /**
   * Tests the constructor.
   *
   * @covers ::__construct
   */
  public function testConstructor(): void {
    $this->assertInstanceOf(FieldPluginBase::class, $this->plugin);
    $this->assertInstanceOf(ViewsConfigField::class, $this->plugin);
  }

  /**
   * Tests the create method.
   *
   * @covers ::create
   */
  public function testCreate(): void {
    $container = new ContainerBuilder();
    $container->set('config.factory', $this->configFactory);

    $configuration = [];
    $plugin_id = 'views_config_field';
    $plugin_definition = [];

    $plugin = ViewsConfigField::create($container, $configuration, $plugin_id, $plugin_definition);

    $this->assertInstanceOf(ViewsConfigField::class, $plugin);
  }

  /**
   * Tests the query method.
   *
   * @covers ::query
   */
  public function testQuery(): void {
    // The query method should not add anything to the query.
    $this->assertNull($this->plugin->query());
  }

  /**
   * Tests the defineOptions method indirectly through plugin options.
   *
   * @covers ::defineOptions
   */
  public function testDefineOptions(): void {
    // Test that the plugin options have the expected default values.
    // Since defineOptions is protected, we test it indirectly through the
    // plugin options that were initialized in setUp().
    $this->assertArrayHasKey('config_name', $this->plugin->options);
    $this->assertArrayHasKey('config_key', $this->plugin->options);
    $this->assertEquals('system.site', $this->plugin->options['config_name']);
    $this->assertEquals('name', $this->plugin->options['config_key']);
  }

  /**
   * Tests the buildOptionsForm method.
   *
   * @covers ::buildOptionsForm
   */
  public function testBuildOptionsForm(): void {
    $form = [];
    $form_state = $this->createMock(FormStateInterface::class);

    $this->plugin->buildOptionsForm($form, $form_state);

    $this->assertArrayHasKey('config_name', $form);
    $this->assertArrayHasKey('config_key', $form);

    // Test config_name field.
    $this->assertEquals('textfield', $form['config_name']['#type']);
    $this->assertEquals('Configuration name', $form['config_name']['#title']);
    $this->assertTrue($form['config_name']['#required']);
    $this->assertEquals('system.site', $form['config_name']['#default_value']);

    // Test config_key field.
    $this->assertEquals('textfield', $form['config_key']['#type']);
    $this->assertEquals('Configuration key', $form['config_key']['#title']);
    $this->assertTrue($form['config_key']['#required']);
    $this->assertEquals('name', $form['config_key']['#default_value']);
  }

  /**
   * Tests the render method with valid configuration.
   *
   * @covers ::render
   */
  public function testRenderWithValidConfig(): void {
    $values = $this->createMock(ResultRow::class);

    $config_data = [
      'name' => 'Test Site',
      'mail' => 'admin@example.com',
    ];

    $this->config->expects($this->once())
      ->method('getRawData')
      ->willReturn($config_data);

    $result = $this->plugin->render($values);

    $this->assertEquals('Test Site', $result);
  }

  /**
   * Tests the render method with nested configuration key.
   *
   * @covers ::render
   */
  public function testRenderWithNestedConfigKey(): void {
    $values = $this->createMock(ResultRow::class);

    // Update plugin options to use nested key.
    $this->plugin->options['config_key'] = 'admin_theme.settings';

    $config_data = [
      'admin_theme' => [
        'settings' => 'claro',
      ],
    ];

    $this->config->expects($this->once())
      ->method('getRawData')
      ->willReturn($config_data);

    $result = $this->plugin->render($values);

    $this->assertEquals('claro', $result);
  }

  /**
   * Tests the render method with missing configuration name.
   *
   * @covers ::render
   */
  public function testRenderWithMissingConfigName(): void {
    $values = $this->createMock(ResultRow::class);

    // Set empty config name.
    $this->plugin->options['config_name'] = '';

    $result = $this->plugin->render($values);

    $this->assertEquals('', $result);
  }

  /**
   * Tests the render method with missing configuration key.
   *
   * @covers ::render
   */
  public function testRenderWithMissingConfigKey(): void {
    $values = $this->createMock(ResultRow::class);

    // Set empty config key.
    $this->plugin->options['config_key'] = '';

    $result = $this->plugin->render($values);

    $this->assertEquals('', $result);
  }

  /**
   * Tests the render method with non-scalar value.
   *
   * @covers ::render
   */
  public function testRenderWithNonScalarValue(): void {
    $values = $this->createMock(ResultRow::class);

    $config_data = [
      'name' => 'Test Site',
      'settings' => [
        'theme' => 'claro',
        'features' => ['admin_toolbar', 'toolbar'],
      ],
    ];

    $this->config->expects($this->once())
      ->method('getRawData')
      ->willReturn($config_data);

    // Update plugin options to use nested key that returns array.
    $this->plugin->options['config_key'] = 'settings';

    $result = $this->plugin->render($values);

    $this->assertIsString($result);
    $this->assertJson($result);
    $this->assertEquals('{"theme":"claro","features":["admin_toolbar","toolbar"]}', $result);
  }

  /**
   * Tests the getNestedValue method with valid keys through render method.
   *
   * @covers ::getNestedValue
   */
  public function testGetNestedValueWithValidKeys(): void {
    $values = $this->createMock(ResultRow::class);

    $config_data = [
      'level1' => [
        'level2' => [
          'level3' => 'value',
        ],
      ],
    ];

    $this->config->expects($this->once())
      ->method('getRawData')
      ->willReturn($config_data);

    // Update plugin options to use nested key.
    $this->plugin->options['config_key'] = 'level1.level2.level3';

    $result = $this->plugin->render($values);

    $this->assertEquals('value', $result);
  }

  /**
   * Tests the getNestedValue method with invalid keys through render method.
   *
   * @covers ::getNestedValue
   */
  public function testGetNestedValueWithInvalidKeys(): void {
    $values = $this->createMock(ResultRow::class);

    $config_data = [
      'level1' => [
        'level2' => 'value',
      ],
    ];

    $this->config->expects($this->once())
      ->method('getRawData')
      ->willReturn($config_data);

    // Update plugin options to use invalid nested key.
    $this->plugin->options['config_key'] = 'level1.level2.nonexistent';

    $result = $this->plugin->render($values);

    // When the nested key doesn't exist, getNestedValue returns NULL,
    // and render() returns 'null' (json_encode(NULL)).
    $this->assertEquals('null', $result);
  }

  /**
   * Tests the getNestedValue method with non-array value through render method.
   *
   * @covers ::getNestedValue
   */
  public function testGetNestedValueWithNonArrayValue(): void {
    $values = $this->createMock(ResultRow::class);

    $config_data = [
      'level1' => 'string_value',
    ];

    $this->config->expects($this->once())
      ->method('getRawData')
      ->willReturn($config_data);

    // Update plugin options to use nested key on non-array value.
    $this->plugin->options['config_key'] = 'level1.level2';

    $result = $this->plugin->render($values);

    // When trying to access nested key on non-array value, getNestedValue
    // returns NULL,
    // and render() returns 'null' (json_encode(NULL)).
    $this->assertEquals('null', $result);
  }

  /**
   * Tests the getNestedValue method with empty keys through render method.
   *
   * @covers ::getNestedValue
   */
  public function testGetNestedValueWithEmptyKeys(): void {
    $values = $this->createMock(ResultRow::class);

    // Update plugin options to use empty key (should return early).
    $this->plugin->options['config_key'] = '';

    $result = $this->plugin->render($values);

    // When config_key is empty, render() returns early with empty string.
    $this->assertEquals('', $result);
  }

}
