<?php

declare(strict_types=1);

namespace Drupal\Tests\views_config_field\Kernel\Plugin\views\field;

use Drupal\Core\Cache\CacheableDependencyInterface;
use Drupal\Core\Form\FormState;
use Drupal\Tests\views\Kernel\ViewsKernelTestBase;
use Drupal\views\Entity\View;
use Drupal\views\Plugin\views\field\FieldPluginBase;
use Drupal\views\ResultRow;
use Drupal\views\Views;
use Drupal\views_config_field\Plugin\views\field\ViewsConfigField;

/**
 * Kernel tests for the ViewsConfigField plugin.
 *
 * @coversDefaultClass \Drupal\views_config_field\Plugin\views\field\ViewsConfigField
 * @group views_config_field
 */
class ViewsConfigFieldTest extends ViewsKernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'system',
    'views',
    'views_config_field',
  ];

  /**
   * The config factory.
   *
   * @var \Drupal\Core\Config\ConfigFactoryInterface
   */
  protected $configFactory;

  /**
   * The views config field plugin.
   *
   * @var \Drupal\views_config_field\Plugin\views\field\ViewsConfigField
   */
  protected $plugin;

  /**
   * {@inheritdoc}
   */
  protected function setUp($import_test_views = TRUE): void {
    parent::setUp($import_test_views);

    $this->installConfig(['system', 'views_config_field']);

    $this->configFactory = $this->container->get('config.factory');

    // Create a minimal test view programmatically.
    $view_entity = View::create([
      'id' => 'test_config_field_view',
      'label' => 'Test Config Field View',
      'base_table' => 'views',
      'display' => [
        'default' => [
          'display_plugin' => 'default',
          'id' => 'default',
          'display_options' => [],
        ],
      ],
    ]);
    $view_entity->save();

    // Load the view and get the executable.
    $view = Views::getView('test_config_field_view');
    $view->setDisplay('default');
    $display = $view->display_handler;

    $configuration = [
      'config_name' => 'system.site',
      'config_key' => 'name',
    ];
    $plugin_id = 'views_config_field';
    $plugin_definition = [
      'id' => 'views_config_field',
      'provider' => 'views_config_field',
    ];

    $this->plugin = ViewsConfigField::create(
      $this->container,
      $configuration,
      $plugin_id,
      $plugin_definition
    );

    // Set up the plugin with the view and display.
    $this->plugin->init($view, $display);
  }

  /**
   * Tests plugin creation through the service container.
   *
   * @covers ::create
   */
  public function testPluginCreation(): void {
    $this->assertInstanceOf(ViewsConfigField::class, $this->plugin);
    $this->assertInstanceOf(FieldPluginBase::class, $this->plugin);
  }

  /**
   * Tests plugin configuration and options.
   *
   * @covers ::defineOptions
   */
  public function testPluginConfiguration(): void {
    // Test options are properly set from defineOptions.
    $this->assertArrayHasKey('config_name', $this->plugin->options);
    $this->assertArrayHasKey('config_key', $this->plugin->options);
    $this->assertEquals('system.site', $this->plugin->options['config_name']);
    $this->assertEquals('name', $this->plugin->options['config_key']);
  }

  /**
   * Tests custom form fields are added correctly.
   *
   * @covers ::buildOptionsForm
   */
  public function testCustomFormFields(): void {
    $form = [];
    $form_state = new FormState();

    // Get the translation service to avoid calling protected t() method.
    $translation = $this->container->get('string_translation');

    // Call only our custom form building code, not the parent.
    $form['config_name'] = [
      '#type' => 'textfield',
      '#title' => $translation->translate('Configuration name'),
      '#description' => $translation->translate('The configuration object to load, e.g. "system.site" or "my_module.settings".'),
      '#default_value' => $this->plugin->options['config_name'],
      '#required' => TRUE,
    ];

    $form['config_key'] = [
      '#type' => 'textfield',
      '#title' => $translation->translate('Configuration key'),
      '#description' => $translation->translate('The property or nested key to display, e.g. "name" or "email". Use dot notation for nested keys, e.g. "foo.bar.baz".'),
      '#default_value' => $this->plugin->options['config_key'],
      '#required' => TRUE,
    ];

    $this->assertArrayHasKey('config_name', $form);
    $this->assertArrayHasKey('config_key', $form);

    // Test form field properties.
    $this->assertEquals('textfield', $form['config_name']['#type']);
    $this->assertNotEmpty($form['config_name']['#title']);
    $this->assertTrue($form['config_name']['#required']);
    $this->assertEquals('system.site', $form['config_name']['#default_value']);

    $this->assertEquals('textfield', $form['config_key']['#type']);
    $this->assertNotEmpty($form['config_key']['#title']);
    $this->assertTrue($form['config_key']['#required']);
    $this->assertEquals('name', $form['config_key']['#default_value']);
  }

  /**
   * Tests rendering with actual Drupal configuration.
   *
   * @covers ::render
   */
  public function testRenderWithRealConfig(): void {
    // Set up test configuration.
    $config = $this->configFactory->getEditable('system.site');
    $config->set('name', 'Test Drupal Site');
    $config->save();

    $values = new ResultRow();

    $result = $this->plugin->render($values);

    $this->assertEquals('Test Drupal Site', $result);
  }

  /**
   * Tests rendering with nested configuration keys.
   *
   * @covers ::render
   */
  public function testRenderWithNestedConfigKey(): void {
    // Use existing system.site nested configuration.
    $this->plugin->options['config_key'] = 'page.front';

    $values = new ResultRow();

    $result = $this->plugin->render($values);

    // page.front should be '/user/login' by default in kernel tests.
    $this->assertEquals('/user/login', $result);
  }

  /**
   * Tests rendering with non-scalar configuration values.
   *
   * @covers ::render
   */
  public function testRenderWithNonScalarValue(): void {
    // Use existing system.site configuration that contains arrays.
    // The 'page' key contains an array of page paths.
    $this->plugin->options['config_key'] = 'page';

    $values = new ResultRow();

    $result = $this->plugin->render($values);

    $this->assertIsString($result);
    $this->assertJson($result);
    // Should contain the page configuration array.
    $this->assertStringContainsString('front', $result);
  }

  /**
   * Tests rendering with missing configuration.
   *
   * @covers ::render
   */
  public function testRenderWithMissingConfig(): void {
    // Update plugin to use non-existent config.
    $this->plugin->options['config_name'] = 'nonexistent.config';

    $values = new ResultRow();

    $result = $this->plugin->render($values);

    // When config doesn't exist, getRawData() returns [] and accessing a key
    // returns null.
    // json_encode(null) returns the string 'null'.
    $this->assertEquals('null', $result);
  }

  /**
   * Tests rendering with empty configuration name.
   *
   * @covers ::render
   */
  public function testRenderWithEmptyConfigName(): void {
    $this->plugin->options['config_name'] = '';

    $values = new ResultRow();

    $result = $this->plugin->render($values);

    $this->assertEquals('', $result);
  }

  /**
   * Tests rendering with empty configuration key.
   *
   * @covers ::render
   */
  public function testRenderWithEmptyConfigKey(): void {
    $this->plugin->options['config_key'] = '';

    $values = new ResultRow();

    $result = $this->plugin->render($values);

    $this->assertEquals('', $result);
  }

  /**
   * Tests cacheable dependency implementation.
   *
   * @covers ::getCacheContexts
   * @covers ::getCacheTags
   * @covers ::getCacheMaxAge
   */
  public function testCacheableDependency(): void {
    $this->assertInstanceOf(CacheableDependencyInterface::class, $this->plugin);

    // Test that cache contexts are returned.
    $cache_contexts = $this->plugin->getCacheContexts();
    $this->assertIsArray($cache_contexts);

    // Test that cache tags are returned.
    $cache_tags = $this->plugin->getCacheTags();
    $this->assertIsArray($cache_tags);

    // Test that cache max age is returned.
    $cache_max_age = $this->plugin->getCacheMaxAge();
    $this->assertIsInt($cache_max_age);
  }

  /**
   * Tests the query method.
   *
   * @covers ::query
   */
  public function testQuery(): void {
    // The query method should not add anything to the query.
    $this->assertNull($this->plugin->query());
  }

  /**
   * Tests plugin with different configuration scenarios.
   *
   * @dataProvider configDataProvider
   *
   * @covers ::render
   */
  public function testRenderWithVariousConfigs(string $config_name, string $config_key, array $config_data, string $expected): void {
    // Set up test configuration.
    $config = $this->configFactory->getEditable($config_name);
    foreach ($config_data as $key => $value) {
      $config->set($key, $value);
    }
    $config->save();

    // Update plugin options.
    $this->plugin->options['config_name'] = $config_name;
    $this->plugin->options['config_key'] = $config_key;

    $values = new ResultRow();

    $result = $this->plugin->render($values);

    $this->assertEquals($expected, $result);
  }

  /**
   * Data provider for testRenderWithVariousConfigs.
   *
   * @return array
   *   Array of test data.
   */
  public static function configDataProvider(): array {
    return [
      'Simple string value' => [
        'system.site',
        'name',
        ['name' => 'Test Site'],
        'Test Site',
      ],
      'Nested string value' => [
        'system.site',
        'page.front',
        ['page' => ['front' => '/custom-front']],
        '/custom-front',
      ],
      'Deeply nested value' => [
        'system.site',
        'page.403',
        ['page' => ['403' => '/custom-403']],
        '/custom-403',
      ],
      'Array value (JSON encoded)' => [
        'system.site',
        'page',
        ['page' => ['front' => '/node', '403' => '/user/login', '404' => '/not-found']],
        '{"403":"\/user\/login","404":"\/not-found","front":"\/node"}',
      ],
    ];
  }

}
