<?php

namespace Drupal\views_csv_source\Plugin\views\filter;

use Drupal\Component\Datetime\TimeInterface;
use Drupal\views\Attribute\ViewsFilter;
use Drupal\views\Plugin\views\filter\Date;
use Drupal\views_csv_source\Plugin\views\ColumnSelectorTrait;
use Drupal\views_csv_source\Query\Select;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Date filter handler for views_csv_source.
 *
 * Extends the core Date filter to provide UI, but overrides query handling
 * to work with CSV text data instead of SQL.
 */
#[ViewsFilter("views_csv_source_filter_datetime")]
class ViewsCsvFilterDatetime extends Date {

  use ColumnSelectorTrait;

  /**
   * Constructs a new ViewsCsvFilterDatetime object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin_id for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Component\Datetime\TimeInterface $time
   *   The time service.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected TimeInterface $time,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('datetime.time')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function query(): void {
    $this->ensureMyTable();

    // The ColumnSelectorTrait stores the selected CSV header in 'key'.
    $field = $this->options['key'];
    if (!$field) {
      return;
    }

    $info = $this->operators();
    if (!empty($info[$this->operator]['method'])) {
      $this->{$info[$this->operator]['method']}($field);
    }
  }

  /**
   * Override opSimple to pass timestamps to the CSV Query.
   *
   * @param string $field
   *   The CSV column key.
   */
  protected function opSimple($field): void {
    // Delegate equality (=, !=) to opBetween.
    if (in_array($this->operator, ['=', '!='])) {
      $this->value['min'] = $this->value['value'];
      $this->value['max'] = $this->value['value'];
      $this->operator = ($this->operator === '!=') ? 'not between' : 'between';
      $this->opBetween($field);
      return;
    }

    $value = $this->value['value'];
    $timestamp = $this->isOffset()
      ? strtotime($value, $this->time->getRequestTime())
      : Select::processDate($value, $this->operator);

    if ($timestamp !== FALSE) {
      // Pass the timestamp. Select.php::verifyCondition will compare this
      // integer against the strtotime() result of the CSV column value.
      $this->query->addWhere($this->options['group'], $field, $timestamp, $this->operator);
    }
  }

  /**
   * Override opBetween to pass timestamp ranges to the CSV Query.
   *
   * @param string $field
   *   The CSV column key.
   */
  protected function opBetween($field): void {
    if (!$this->value['min'] || !$this->value['max']) {
      return;
    }

    $is_offset = $this->isOffset();
    $min_value = $is_offset
      ? strtotime($this->value['min'], $this->time->getRequestTime())
      : Select::processDate($this->value['min'], '>=');
    $max_value = $is_offset
      ? strtotime($this->value['max'], $this->time->getRequestTime())
      : Select::processDate($this->value['max'], '<=');

    // IMPORTANT: We must pass an indexed array [$min, $max] here, not an
    // associative array ['min' => $min]. The Select class decodes internal
    // JSON using json_decode($json, FALSE), which converts associative arrays
    // into stdClass objects. The Select::verifyConditionBetween method fails
    // if it receives an object instead of an array. Indexed arrays survive
    // the decode process as arrays.
    if ($this->operator === 'between') {
      $this->query->addWhere($this->options['group'], $field, [$min_value, $max_value], 'between');
    }
    else {
      $this->query->addWhere($this->options['group'], $field, [$min_value, $max_value], $this->operator);
    }
  }

  /**
   * Determines if the current value is of type 'offset'.
   *
   * @return bool
   *   Returns true if the value type is 'offset', otherwise false.
   */
  protected function isOffset(): bool {
    return !empty($this->value['type']) && $this->value['type'] === 'offset';
  }

}
