<?php

namespace Drupal\views_cumulative_field\Plugin\views\field;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsField;
use Drupal\views\Plugin\views\field\NumericField;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Field handler for cumulative field calculations.
 *
 * @ingroup views_field_handlers
 */
#[ViewsField("field_cumulative_field")]
class CumulativeField extends NumericField {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Tracks cumulative sum for PHP-based calculation.
   *
   * @var float
   */
  private $cumulativeSum = 0;

  /**
   * {@inheritdoc}
   */
  public function usesGroupBy(): bool {
    return FALSE;
  }

  /**
   * Views Cumulative Field constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function query(): void {
    // Ensure the table is added.
    $this->ensureMyTable();

    // Aggregation (GROUP BY) conflicts with window functions in the SELECT list
    // because Views automatically adds the window function alias to the
    // GROUP BY clause, triggering SQL error 4015.
    // Therefore, if aggregation is enabled, we must fall back to PHP
    // calculation.
    $is_aggregated = $this->displayHandler->useGroupBy();

    if ($this->options['summation_method'] === 'database' && !$is_aggregated) {
      $this->addDatabaseCumulativeField();
    }
    else {
      // Initialize a placeholder for PHP-based processing.
      // This ensures the field exists in the row for rendering.
      $this->additional_fields['cumulative_field_data'] = 0;
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions(): array {
    $options = parent::defineOptions();
    $options['data_field'] = ['default' => NULL];
    $options['summation_method'] = ['default' => 'php'];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);
    $field_options = $this->displayHandler->getFieldLabels();

    // Prevent recursion by removing self from options.
    if (isset($field_options[$this->options['id']])) {
      unset($field_options[$this->options['id']]);
    }

    $form['summation_method'] = [
      '#type' => 'radios',
      '#title' => $this->t('Summation Method'),
      '#options' => [
        'php' => $this->t('PHP'),
        'database' => $this->t('Database'),
      ],
      '#default_value' => $this->options['summation_method'] ?? 'php',
      '#description' => $this->t('Select how to calculate cumulative values. Database method is more efficient for large datasets but will automatically fallback to PHP if Views Aggregation is enabled.'),
      '#weight' => -11,
    ];
    $form['data_field'] = [
      '#type' => 'radios',
      '#title' => $this->t('Data Field'),
      '#options' => $field_options,
      '#default_value' => $this->options['data_field'],
      '#description' => $this->t('Select the field for which to calculate the cumulative value.'),
      '#weight' => -10,
    ];
  }

  /**
   * Adds a database-level cumulative field calculation.
   */
  protected function addDatabaseCumulativeField(): void {
    $field = $this->options['data_field'];
    if (empty($field)) {
      return;
    }

    // Get the handler for the data field.
    $field_handler = $this->displayHandler->getHandler('field', $field);
    if (!$field_handler) {
      return;
    }

    // Resolve Aliases for the Data Field.
    $field_table_alias = $this->query->ensureTable($field_handler->table, $field_handler->relationship);
    $field_name = $field_handler->realField ?? $field_handler->field;

    if (!$field_table_alias) {
      return;
    }

    $orderby_parts = $this->getOrderByClause();
    if (empty($orderby_parts)) {
      return;
    }

    $orderby_clause = implode(', ', $orderby_parts);
    $formula = "SUM($field_table_alias.$field_name) OVER (ORDER BY $orderby_clause)";

    $this->field_alias = $this->query->addField(
      NULL,
      $formula,
      $this->field . '_cumulative'
    );

    $this->addAdditionalFields();
  }

  /**
   * Generates the ORDER BY clause for the window function.
   *
   * @return array
   *   An array of SQL order by strings.
   *
   * @throws \Drupal\Component\Plugin\Exception\PluginNotFoundException
   */
  protected function getOrderByClause(): array {
    $orderby_parts = [];
    $sorts = $this->displayHandler->getHandlers('sort');

    foreach ($sorts as $sort) {
      if (empty($sort->options['order'])) {
        continue;
      }

      // Ensure the sort table is available in the query.
      $sort_table_alias = $this->query->ensureTable($sort->table, $sort->relationship);
      $sort_field_name = $sort->realField ?? $sort->field;
      $sort_direction = strtoupper($sort->options['order']);

      // Only add valid SQL sort fields.
      if ($sort_table_alias && $sort_field_name) {
        $orderby_parts[] = "$sort_table_alias.$sort_field_name $sort_direction";
      }
    }

    // Fallback sort if no specific sorts exist (sort by ID DESC).
    if (empty($orderby_parts)) {
      $id_field = NULL;
      $base_entity_type = $this->view->storage->get('base_entity_type');

      // Try to get ID from entity definition.
      if ($base_entity_type && $this->entityTypeManager->hasDefinition($base_entity_type)) {
        $id_field = $this->entityTypeManager->getDefinition($base_entity_type)->getKey('id');
      }

      // Try to get ID from the View base field (for non-entities).
      if (empty($id_field)) {
        $id_field = $this->view->storage->get('base_field');
      }

      // Apply the fallback sort.
      if ($id_field) {
        $base_table_alias = $this->query->ensureTable($this->view->storage->get('base_table'), $this->relationship);
        // "DESC" assumes higher IDs are newer, matching the user
        // expectation of "Newest First".
        $orderby_parts[] = "$base_table_alias.$id_field DESC";
      }
    }

    return $orderby_parts;
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ResultRow $values, $field = NULL): int|float|string {
    $is_aggregated = $this->displayHandler->useGroupBy();

    // Use database value only if configured and aggregation is off.
    if ($this->options['summation_method'] === 'database' && !$is_aggregated) {
      return $this->getValueFromDatabase($values);
    }

    // Fallback to PHP for aggregated views or if explicitly selected.
    return $this->getValueFromPhp($values);
  }

  /**
   * Gets the cumulative value using database calculation.
   *
   * @param \Drupal\views\ResultRow $values
   *   The result row object.
   *
   * @return int|float
   *   The cumulative sum value.
   */
  protected function getValueFromDatabase(ResultRow $values): int|float {
    return isset($values->{$this->field_alias}) ? (float) $values->{$this->field_alias} : 0;
  }

  /**
   * Gets the cumulative value using PHP calculation.
   *
   * @param \Drupal\views\ResultRow $values
   *   The result row object.
   *
   * @return int|float
   *   The cumulative sum value.
   */
  protected function getValueFromPhp(ResultRow $values): int|float {
    $field_id = $this->options['data_field'];
    if (empty($field_id)) {
      return 0;
    }

    $handler = $this->displayHandler->getHandler('field', $field_id);
    if (!$handler) {
      return $this->cumulativeSum;
    }

    $is_aggregated = $this->displayHandler->useGroupBy();

    if ($is_aggregated) {
      // When aggregated, the row already contains the SUM/AVG/COUNT
      // calculated by the database. We simply extract that value.
      $data = (float) $handler->getValue($values);
    }
    else {
      // Non-aggregated fallback logic (Entities, Commerce, Rewrites).
      $data = $this->getNonAggregatedValue($values, $field_id);
    }

    $this->cumulativeSum += $data;
    return $this->cumulativeSum;
  }

  /**
   * Helper to retrieve value for non-aggregated rows.
   *
   * @param \Drupal\views\ResultRow $values
   *   The result row.
   * @param string $field_id
   *   The field ID to retrieve.
   *
   * @return int|float
   *   The extracted value.
   */
  protected function getNonAggregatedValue(ResultRow $values, string $field_id): int|float {
    $data = 0;
    $field_type = $this->getFieldType($field_id);
    $rewritten = $this->getRewriteStatus($field_id);
    $relationship = $this->getFieldRelationship($field_id);

    // Try entity access.
    if ($field_type !== 'undefined') {
      $entity = $relationship
        ? $this->getRelationshipEntity($values, $field_id, $relationship)
        : ($values->_entity ?? NULL);

      if ($entity instanceof EntityInterface) {
        if ($rewritten) {
          if (is_numeric($rewritten)) {
            $data = (float) $rewritten;
          }
          else {
            $render_output = $this->displayHandler->getHandler('field', $field_id)->advancedRender($values);
            $data = (float) (is_array($render_output) ? current($render_output) : $render_output);
          }
        }
        else {
          $handler = $this->displayHandler->getHandler('field', $field_id);
          $field_base = $handler->field;

          if ($entity->hasField($field_base)) {
            $raw_val = $entity->get($field_base)->getValue();
            $data = (float) ($raw_val[0]['value'] ?? 0);
          }
          elseif ($field_type === 'commerce_price_default' || $field_type === 'commerce_product_variation') {
            if ($entity->hasField('price')) {
              $raw_val = $entity->get('price')->getValue();
              $data = (float) ($raw_val[0]['number'] ?? 0);
            }
          }
        }
        if ($data != 0) {
          return $data;
        }
      }
    }

    // Fallback to standard handler value.
    $handler = $this->displayHandler->getHandler('field', $field_id);
    $data = (float) $handler->getValue($values);

    if ($rewritten) {
      $render_output = $handler->advancedRender($values);
      $data = (float) (is_array($render_output) ? current($render_output) : $render_output);
    }

    return $data;
  }

  /**
   * Determines the field type.
   *
   * @param string $field
   *   The name of the field for which to retrieve the type.
   *
   * @return string
   *   The field type of the provided field.
   */
  protected function getFieldType(string $field): string {
    $field_handler = $this->displayHandler->getHandler('field', $field)->options ?? NULL;
    return $field_handler['type'] ?? 'undefined';
  }

  /**
   * Determines if the field comes from a relationship.
   *
   * @param string $field
   *   The name of the field for which to retrieve the relationship.
   *
   * @return string|null
   *   The relationship needed to join tables to retrieve the field data.
   */
  protected function getFieldRelationship(string $field): ?string {
    $field_handler = $this->displayHandler->getHandler('field', $field)->options ?? NULL;
    if (!empty($field_handler['relationship']) && $field_handler['relationship'] !== 'none') {
      return $field_handler['relationship'];
    }
    return NULL;
  }

  /**
   * Determines whether the field is rewritten/altered.
   *
   * @param string $field
   *   The name of the field to retrieve the rewrite status.
   *
   * @return string|null
   *   The rewrite text for the provided field, or NULL if not rewritten.
   */
  protected function getRewriteStatus(string $field): ?string {
    $field_handler = $this->displayHandler->getHandler('field', $field)->options ?? NULL;
    if (!empty($field_handler['alter']['alter_text']) && !empty($field_handler['alter']['text'])) {
      return $field_handler['alter']['text'];
    }
    return NULL;
  }

  /**
   * Retrieves relationship entity for given values.
   *
   * @param \Drupal\views\ResultRow $values
   *   The result row object.
   * @param string $field
   *   The name of the field to retrieve the relationship entity.
   * @param string $relationship
   *   The name of the relationship to retrieve.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   An entity representing the relationship, or null if not found.
   */
  protected function getRelationshipEntity(ResultRow $values, string $field, string $relationship): ?EntityInterface {
    $relationship_entities = $values->_relationship_entities ?? [];
    return $relationship_entities[$relationship] ?? NULL;
  }

}
