<?php

namespace Drupal\views_cumulative_field\Plugin\views\field;

use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsField;
use Drupal\views\Plugin\views\field\NumericField;
use Drupal\views\ResultRow;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Field handler for cumulative total field calculations.
 *
 * Displays the grand total of the selected field for the current result set
 * on every row.
 *
 * @ingroup views_field_handlers
 */
#[ViewsField("field_cumulative_total")]
class CumulativeTotalField extends NumericField {

  /**
   * The entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * Stores the calculated grand total for PHP-based calculation.
   *
   * @var float|null
   */
  private $totalSum = NULL;

  /**
   * {@inheritdoc}
   */
  public function usesGroupBy(): bool {
    return FALSE;
  }

  /**
   * Views Cumulative Total Field constructor.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, EntityTypeManagerInterface $entity_type_manager) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->entityTypeManager = $entity_type_manager;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition): static {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function query(): void {
    // Ensure the table is added.
    $this->ensureMyTable();

    // Check if Views aggregation is enabled.
    $is_aggregated = $this->displayHandler->useGroupBy();

    // Use database method only if selected and aggregation is disabled.
    // If aggregation is on, we fall back to PHP to avoid SQL window function
    // conflicts with the GROUP BY clause.
    if ($this->options['summation_method'] === 'database' && !$is_aggregated) {
      $this->addDatabaseCumulativeField();
    }
    else {
      // For PHP method, initialize the field data.
      $this->additional_fields['cumulative_total_data'] = 0;
    }
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions(): array {
    $options = parent::defineOptions();
    $options['data_field'] = ['default' => NULL];
    $options['summation_method'] = ['default' => 'database'];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);
    $field_options = $this->displayHandler->getFieldLabels();

    // Filter self out of options to prevent recursion.
    if (isset($field_options[$this->options['id']])) {
      unset($field_options[$this->options['id']]);
    }

    $form['summation_method'] = [
      '#type' => 'radios',
      '#title' => $this->t('Summation Method'),
      '#options' => [
        'php' => $this->t('PHP'),
        'database' => $this->t('Database'),
      ],
      '#default_value' => $this->options['summation_method'] ?? 'database',
      '#description' => $this->t('Select how to calculate the total. Database method is more efficient for large datasets but will automatically fallback to PHP if Views Aggregation is enabled.'),
      '#weight' => -11,
    ];
    $form['data_field'] = [
      '#type' => 'radios',
      '#title' => $this->t('Data Field'),
      '#options' => $field_options,
      '#default_value' => $this->options['data_field'],
      '#description' => $this->t('Select the field to sum.'),
      '#weight' => -10,
    ];
  }

  /**
   * Adds a database-level total calculation.
   */
  protected function addDatabaseCumulativeField(): void {
    $field = $this->options['data_field'];
    if (empty($field)) {
      return;
    }

    // Get the handler for the data field.
    $field_handler = $this->displayHandler->getHandler('field', $field);
    if (!$field_handler) {
      return;
    }

    // Resolve aliases for the Data Field.
    $field_table_alias = $this->query->ensureTable($field_handler->table, $field_handler->relationship);
    $field_name = $field_handler->realField ?? $field_handler->field;

    if (!$field_table_alias) {
      return;
    }

    // Construct the window function for Grand Total.
    // Unlike the cumulative field, we do NOT use an ORDER BY clause.
    // SUM(field) OVER () calculates the sum of the entire partition.
    $formula = "SUM($field_table_alias.$field_name) OVER ()";

    // Add the formula.
    $this->field_alias = $this->query->addField(
      NULL,
      $formula,
      $this->field . '_total'
    );

    $this->addAdditionalFields();
  }

  /**
   * {@inheritdoc}
   */
  public function getValue(ResultRow $values, $field = NULL): int|float|string {
    $is_aggregated = $this->displayHandler->useGroupBy();

    // For database method, return the pre-calculated value.
    if ($this->options['summation_method'] === 'database' && !$is_aggregated) {
      return $this->getValueFromDatabase($values);
    }

    // For PHP method (or fallback), return the calculated total.
    return $this->getValueFromPhp($values);
  }

  /**
   * Gets the total value using database calculation.
   *
   * @param \Drupal\views\ResultRow $values
   *   The result row object.
   *
   * @return int|float
   *   The total sum value.
   */
  protected function getValueFromDatabase(ResultRow $values): int|float {
    if (isset($values->{$this->field_alias})) {
      return (float) $values->{$this->field_alias};
    }
    return 0;
  }

  /**
   * Gets the total value using PHP calculation.
   *
   * @param \Drupal\views\ResultRow $values
   *   The result row object.
   *
   * @return int|float
   *   The total sum value.
   */
  protected function getValueFromPhp(ResultRow $values): int|float {
    // If we have calculated the total for this view execution, return it.
    if ($this->totalSum !== NULL) {
      return $this->totalSum;
    }

    // Calculate the total by iterating over the entire result set once.
    $total = 0;

    // Note: $this->view->result contains all rows for the current page.
    foreach ($this->view->result as $row) {
      $total += $this->calculateSingleRowValue($row);
    }

    $this->totalSum = $total;
    return $this->totalSum;
  }

  /**
   * Calculates the value for a specific row.
   *
   * @param \Drupal\views\ResultRow $row
   *   The result row object.
   *
   * @return int|float
   *   The numeric value of the data field for this row.
   */
  protected function calculateSingleRowValue(ResultRow $row): int|float {
    $field = $this->options['data_field'];
    if (empty($field)) {
      return 0;
    }

    $field_type = $this->getFieldType($field);
    $rewritten = $this->getRewriteStatus($field);
    $data = 0;
    $is_aggregated = $this->displayHandler->useGroupBy();

    if (!$is_aggregated && $field_type !== 'undefined') {
      $relationship = $this->getFieldRelationship($field);
      $entity = $relationship
        ? $this->getRelationshipEntity($row, $field, $relationship)
        : ($row->_entity ?? NULL);

      if ($entity instanceof EntityInterface) {
        if ($rewritten) {
          if (is_numeric($rewritten)) {
            $data = $rewritten;
          }
          else {
            $render = $this->displayHandler->getHandler('field', $field)->advancedRender($row);
            $data = is_array($render) ? current($render) : $render;
          }
        }
        else {
          $field_handler = $this->displayHandler->getHandler('field', $field);
          $field_base = $field_handler->field;

          if ($entity->hasField($field_base)) {
            $val = $entity->get($field_base)->getValue();
            $data = $val[0]['value'] ?? 0;
          }
          elseif ($field_type === 'commerce_price_default' || $field_type === 'commerce_product_variation') {
            if ($entity->hasField('price')) {
              $val = $entity->get('price')->getValue();
              $data = $val[0]['number'] ?? 0;
            }
          }
        }
      }
    }
    else {
      // Aggregated fallback or undefined field types.
      $handler = $this->displayHandler->getHandler('field', $field);
      $data = $handler->getValue($row) ?? 0;

      if ($rewritten) {
        $render = $handler->advancedRender($row);
        $data = is_array($render) ? current($render) : $render;
      }
    }

    return (float) preg_replace('/[^0-9.-]/', '', (string) $data);
  }

  /**
   * Determines the field type.
   *
   * @param string $field
   *   The name of the field for which to retrieve the type.
   *
   * @return string
   *   The field type of the provided field.
   */
  protected function getFieldType(string $field): string {
    $field_handler = $this->displayHandler->getHandler('field', $field)->options ?? NULL;
    return $field_handler['type'] ?? 'undefined';
  }

  /**
   * Determines if the field comes from a relationship.
   *
   * @param string $field
   *   The name of the field for which to retrieve the relationship.
   *
   * @return string|null
   *   The relationship needed to join tables to retrieve the field data.
   */
  protected function getFieldRelationship(string $field): ?string {
    $field_handler = $this->displayHandler->getHandler('field', $field)->options ?? NULL;
    if (!empty($field_handler['relationship']) && $field_handler['relationship'] !== 'none') {
      return $field_handler['relationship'];
    }
    return NULL;
  }

  /**
   * Determines whether the field is rewritten/altered.
   *
   * @param string $field
   *   The name of the field to retrieve the rewrite status.
   *
   * @return string|null
   *   The rewrite text for the provided field, or NULL if not rewritten.
   */
  protected function getRewriteStatus(string $field): ?string {
    $field_handler = $this->displayHandler->getHandler('field', $field)->options ?? NULL;
    if (!empty($field_handler['alter']['alter_text']) && !empty($field_handler['alter']['text'])) {
      return $field_handler['alter']['text'];
    }
    return NULL;
  }

  /**
   * Retrieves relationship entity for given values.
   *
   * @param \Drupal\views\ResultRow $values
   *   The result row object.
   * @param string $field
   *   The name of the field to retrieve the relationship entity.
   * @param string $relationship
   *   The name of the relationship to retrieve.
   *
   * @return \Drupal\Core\Entity\EntityInterface|null
   *   An entity representing the relationship, or null if not found.
   */
  protected function getRelationshipEntity(ResultRow $values, string $field, string $relationship): ?EntityInterface {
    $relationship_entities = $values->_relationship_entities ?? [];
    return $relationship_entities[$relationship] ?? NULL;
  }

}
