<?php

namespace Drupal\views_data_export_tcpdf\Encoder;

use Drupal\Component\Serialization\Exception\InvalidDataTypeException;
use Drupal\Component\Utility\Html;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\file\FileInterface;
use Symfony\Component\DependencyInjection\Attribute\AutowireServiceTag;
use Symfony\Component\Serializer\Encoder\EncoderInterface;
use Symfony\Component\Serializer\Encoder\NormalizationAwareInterface;
use Drupal\views_data_export_tcpdf\Pdf\CustomPdf;

/**
 * Adds PDF encoder support for the Serialization API.
 */
#[AutowireServiceTag('encoder')]
class Pdf implements EncoderInterface, NormalizationAwareInterface {

  /**
   * The format supported by this encoder.
   *
   * @var string
   */
  protected static string $format = 'pdf';

  public function __construct(
    private readonly FileSystemInterface $fileSystem,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {}

  /**
   * {@inheritdoc}
   */
  public function encode(mixed $data, string $format, array $context = []): string {
    $data = match (gettype($data)) {
      'array' => $data,
      'object' => (array) $data,
      default => [$data],
    };

    try {
      $pdf = new CustomPdf(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, TRUE, 'UTF-8', FALSE);

      $pdf_options = $context['views_style_plugin']->options['pdf_settings'] ?? [];
      $metadata = $pdf_options['metadata'] ?? [];
      $header_options = $pdf_options['header'] ?? [];
      $footer_options = $pdf_options['footer'] ?? [];
      $logo_fid = $header_options['logo'] ?? ($pdf_options['logo'] ?? NULL);

      $pdf->SetCreator($metadata['creator'] ?? PDF_CREATOR);
      $pdf->SetAuthor($metadata['creator'] ?? 'Drupal');
      $pdf->SetTitle($metadata['title'] ?? 'Export');
      $pdf->SetSubject($metadata['subject'] ?? '');
      $pdf->SetKeywords($metadata['keywords'] ?? '');

      $view = $context['views_style_plugin']->view ?? NULL;
      $view_title = $view?->getTitle();

      $logo_path = NULL;
      if (!empty($logo_fid)) {
        $logo_id = is_array($logo_fid) ? reset($logo_fid) : $logo_fid;
        if ($logo_id) {
          $file_storage = $this->entityTypeManager->getStorage('file');
          $logo_file = $file_storage->load($logo_id);
          if ($logo_file instanceof FileInterface) {
            $resolved_logo_path = $this->fileSystem->realpath($logo_file->getFileUri());
            if ($resolved_logo_path && file_exists($resolved_logo_path)) {
              $logo_path = $resolved_logo_path;
            }
          }
        }
      }

      $header_title = trim((string) ($header_options['title'] ?? '')) ?: (string) ($view_title ?? '');
      $header_lines = array_values(array_filter([
        trim((string) ($header_options['subtitle'] ?? '')),
        implode(' | ', array_filter([
          trim((string) ($header_options['phone'] ?? '')),
          trim((string) ($header_options['email'] ?? '')),
          trim((string) ($header_options['website'] ?? '')),
        ])),
        trim((string) ($header_options['address'] ?? '')),
        trim((string) ($header_options['notes'] ?? '')),
      ]));

      $header_enabled = $header_title !== '' || !empty($header_lines) || !empty($logo_path);

      if ($header_enabled) {
        $pdf->setPrintHeader(TRUE);
        $pdf->setHeaderConfig([
          'title' => $header_title,
          'lines' => $header_lines,
          'logo_path' => $logo_path,
        ]);
      }
      else {
        $pdf->setPrintHeader(FALSE);
      }

      $footer_values = [
        'left' => trim((string) ($footer_options['left'] ?? '')),
        'center' => trim((string) ($footer_options['center'] ?? '')),
        'right' => trim((string) ($footer_options['right'] ?? '')),
      ];
      $footer_enabled = (bool) array_filter($footer_values);

      if ($footer_enabled) {
        $pdf->setPrintFooter(TRUE);
        $pdf->setFooterConfig($footer_values);
      }
      else {
        $pdf->setPrintFooter(FALSE);
      }
      $pdf->SetDefaultMonospacedFont(PDF_FONT_MONOSPACED);
      $top_margin = $header_enabled ? 42 : PDF_MARGIN_LEFT;
      $bottom_margin = $footer_enabled ? 25 : PDF_MARGIN_BOTTOM;
      $pdf->SetMargins(PDF_MARGIN_LEFT, $top_margin, PDF_MARGIN_RIGHT);
      $pdf->SetAutoPageBreak(TRUE, $bottom_margin);
      $pdf->setImageScale(PDF_IMAGE_SCALE_RATIO);
      $pdf->AddPage('L');

      $header_map = $this->extractHeaders($data, $context);
      $table_markup = $this->buildTableHtml($header_map, $data, $view_title);

      $pdf->writeHTML($table_markup, TRUE, FALSE, TRUE, FALSE, '');

      return $pdf->Output('', 'S');
    }
    catch (\Exception $e) {
      throw new InvalidDataTypeException($e->getMessage(), $e->getCode(), $e);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function supportsEncoding(string $format): bool {
    return $format === static::$format;
  }

  /**
   * Builds the styled HTML table that renders the export rows.
   */
  protected function buildTableHtml(array $header_map, array $rows, ?string $view_title): string {
    $column_keys = array_keys($header_map);
    if (empty($column_keys) && !empty($rows)) {
      $first_row = reset($rows);
      $column_keys = array_keys($first_row);
      $header_map = array_combine($column_keys, $column_keys);
    }

    $column_count = max(1, count($column_keys));
    $row_total = count($rows);
    $generated_at = (new \DateTimeImmutable('now'))->format('d/m/Y H:i');

    $html = $this->getPdfStyles();
    $html .= '<table class="pdf-table" cellpadding="0" cellspacing="0">';

    if (!empty($header_map)) {
      $html .= '<thead><tr>';
      foreach ($header_map as $label) {
        $html .= '<th>' . Html::escape((string) $label) . '</th>';
      }
      $html .= '</tr></thead>';
    }

    $html .= '<tbody>';

    if (!empty($rows)) {
      foreach ($rows as $index => $row) {
        $row_class = $index % 2 === 0 ? 'even' : 'odd';
        $html .= '<tr class="' . $row_class . '">';
        foreach ($column_keys as $column_key) {
          $cell_value = $row[$column_key] ?? '';
          $html .= '<td>' . $this->formatCellValue($cell_value) . '</td>';
        }
        $html .= '</tr>';
      }
    }
    else {
      $html .= '<tr><td colspan="' . $column_count . '" class="empty-state">' . Html::escape('No data available for this export.') . '</td></tr>';
    }

    $html .= '</tbody></table>';

    $summary_text = sprintf('Generated on %s • Total: %d record(s)', $generated_at, $row_total);
    $html .= '<div class="summary-bar">' . Html::escape($summary_text) . '</div>';

    return $html;
  }

  /**
   * Formats a cell value for safe HTML rendering inside the PDF.
   */
  protected function formatCellValue(mixed $value): string {
    if (is_array($value)) {
      $parts = [];
      foreach ($value as $item) {
        $normalized = $this->normalizeScalar($item);
        if ($normalized !== '') {
          $parts[] = $normalized;
        }
      }
      $value = implode(', ', $parts);
    }
    else {
      $value = $this->normalizeScalar($value);
    }

    $value = trim(strip_tags($value));
    return Html::escape($value);
  }

  /**
   * Normalizes different value types into a printable scalar string.
   */
  protected function normalizeScalar(mixed $value): string {
    if ($value instanceof \Stringable) {
      return (string) $value;
    }
    if (is_bool($value)) {
      return $value ? 'Yes' : 'No';
    }
    if (is_object($value)) {
      return method_exists($value, '__toString') ? (string) $value : (json_encode($value, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES) ?: '');
    }
    if ($value === NULL) {
      return '';
    }

    return (string) $value;
  }

  /**
   * Extracts headers from the data array.
   *
   * @param array $data
   *   The data array.
   * @param array $context
   *   The context options array.
   *
   * @return array
   *   An array of headers.
   */
  protected function extractHeaders(array $data, array $context): array {
    $header_map = [];
    if ($first_row = reset($data)) {
      $keys = array_keys($first_row);

      if (!empty($context['header']) && is_array($context['header'])) {
        foreach ($context['header'] as $delta => $label) {
          $key = $keys[$delta] ?? (string) $delta;
          $header_map[$key] = $label;
        }
      }
      elseif (!empty($context['views_style_plugin'])) {
        $view = $context['views_style_plugin']->view;
        $fields = $view->field;
        foreach ($keys as $key) {
          $header_map[$key] = $fields[$key]->options['label'] ?? $key;
        }
      }
      else {
        foreach ($keys as $key) {
          $header_map[$key] = $key;
        }
      }
    }

    return $header_map;
  }

  /**
   * Returns the CSS styles for the PDF table.
   */
  protected function getPdfStyles(): string {
    return <<<CSS
  <style>
    body { font-family: helvetica, sans-serif; }

    .pdf-table {
      width: 100%;
      border-collapse: collapse;
      margin: 0;
      font-size: 8pt;
      color: #1a202c;
    }

    .pdf-table thead th {
      background: linear-gradient(180deg, #0f4c81 0%, #0a355c 100%);
      color: #ffffff;
      font-weight: bold;
      font-size: 7.5pt;
      text-transform: uppercase;
      letter-spacing: 0.05em;
      padding: 10px 8px;
      text-align: left;
      border-bottom: 2px solid #0a355c;
    }

    .pdf-table tbody td {
      padding: 8px;
      border-bottom: 1px solid #e2e8f0;
      font-size: 8pt;
      vertical-align: top;
    }

    .pdf-table tbody tr.even {
      background-color: #f7fafc;
    }

    .pdf-table tbody tr.odd {
      background-color: #ffffff;
    }

    .pdf-table tbody tr:last-child td {
      border-bottom: 2px solid #cbd5e0;
    }

    .empty-state {
      text-align: center;
      font-style: italic;
      color: #a0aec0;
      padding: 30px !important;
    }

    .summary-bar {
      margin-top: 12px;
      padding: 8px 12px;
      background-color: #f7fafc;
      border-left: 3px solid #0f4c81;
      font-size: 7pt;
      color: #4a5568;
    }
  </style>
CSS;
  }

}
