<?php

namespace Drupal\views_data_export_tcpdf\EventSubscriber;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\views_data_export_tcpdf\Plugin\views\display\PdfDataExport;
use Symfony\Component\EventDispatcher\EventSubscriberInterface;
use Symfony\Component\HttpKernel\Event\RequestEvent;
use Symfony\Component\HttpKernel\KernelEvents;

/**
 * Handles redirect logic for PDF export routes.
 */
class ExportRedirectSubscriber implements EventSubscriberInterface {

  /**
   * The allowed PDF content types.
   *
   * @var array
   */
  protected readonly array $pdfContentTypes;

  public function __construct(
    private readonly RouteMatchInterface $routeMatch,
    private readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    $this->pdfContentTypes = ['pdf'];
  }

  /**
   * {@inheritdoc}
   */
  public static function getSubscribedEvents(): array {
    return [
      KernelEvents::REQUEST => ['exportRedirect', -64],
    ];
  }

  /**
   * Redirects to PDF export if all necessary route parameters are present.
   *
   * @param \Symfony\Component\HttpKernel\Event\RequestEvent $event
   *   The request event.
   */
  public function exportRedirect(RequestEvent $event): void {
    $request = $event->getRequest();
    // Determine if this is a PDF form route.
    $view_id = $this->routeMatch->getParameter('view_id');
    $display_id = $this->routeMatch->getParameter('display_id');
    if (isset($view_id, $display_id) && $this->routeMatch->getRouteName() === "view.$view_id.$display_id.export") {
      $args = $this->getViewArgs($this->routeMatch);
      $pdf_file_id = $this->routeMatch->getParameter('_pdf_file');
      $pdf_file = $this->fileStorage->load($pdf_file_id);
      if ($pdf_file && in_array($request->getFormat($pdf_file->getMimeType()), $this->pdfContentTypes, TRUE)) {
        $args['_pdf_file'] = $pdf_file;
        $event->setResponse(PdfDataExport::buildResponse($view_id, $display_id, $args));
      }
    }
  }

  /**
   * Extracts view arguments from the route.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface|null $route_match
   *   The route match.
   *
   * @return array
   *   The view arguments.
   */
  protected function getViewArgs(?RouteMatchInterface $route_match = NULL): array {
    if (!isset($route_match)) {
      return [];
    }

    $route = $route_match->getRouteObject();
    $map = $route->getOption('_view_argument_map') ?? [];
    $args = [];

    foreach ($map as $attribute => $parameter_name) {
      $attribute = $map[$attribute] ?? $attribute;
      $arg = $route_match->getRawParameter($attribute) ?? $route_match->getParameter($attribute);

      if (isset($arg)) {
        $args[] = $arg;
      }
    }

    return $args;
  }

}
