<?php

namespace Drupal\views_data_export_tcpdf\Form;

use Drupal\Component\Utility\Environment;
use Drupal\Core\DependencyInjection\ContainerInjectionInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\File\FileSystemInterface;
use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\file\FileInterface;
use Drupal\file\FileStorageInterface;
use Drupal\file\Upload\FileUploadHandler;
use Drupal\views_data_export_tcpdf\Plugin\views\display\PdfDataExport;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Provides a form for configuring PDF export options.
 */
class PdfExportForm extends FormBase implements ContainerInjectionInterface {

  /**
   * Uploaded file entity.
   *
   * @var \Drupal\file\FileInterface|null
   */
  protected ?FileInterface $file = NULL;

  /**
   * The allowed PDF mime types.
   *
   * @var array
   */
  protected array $pdfMimeTypes = [
    'application/vnd.pdf',
    'application/pdf',
  ];

  public function __construct(
    protected FileSystemInterface $fileSystem,
    protected EntityTypeManagerInterface $entityTypeManager,
    protected FileUploadHandler $fileUploadHandler,
    protected FileStorageInterface $fileStorage,
  ) {}

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('file_system'),
      $container->get('entity_type.manager'),
      $container->get('file.upload_handler'),
      $container->get('entity_type.manager')->getStorage('file'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'pdf_export_form';
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state, $view_id = NULL, $display_id = NULL, ?RouteMatchInterface $route_match = NULL): array {
    $pdf_file_id = $route_match->getParameter('_pdf_file');
    $pdf_file = $this->fileStorage->load($pdf_file_id);
    if ($pdf_file && in_array($pdf_file->getMimeType(), $this->pdfMimeTypes, TRUE)) {
      $this->file = $pdf_file;
    }

    $form['export_external'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Export to existing PDF'),
      '#default_value' => TRUE,
      '#weight' => -1,
    ];

    $validators = [
      'file_validate_extensions' => ['pdf pdfx'],
      'file_validate_size' => [Environment::getUploadMaxSize()],
    ];

    if ($pdf_file == '-1' || is_null($this->file)) {
      $form['pdf_file'] = [
        '#type' => 'file',
        '#title' => $this->t('PDF File'),
        '#description' => [
          '#theme' => 'file_upload_help',
          '#description' => $this->t('An pdf file'),
          '#upload_validators' => $validators,
        ],
        '#upload_validators' => $validators,
      ];
    }
    else {
      $form['pdf_file'] = [
        '#type' => 'entity_autocomplete',
        '#target_type' => 'file',
        '#default_value' => $this->file,
        '#disabled' => TRUE,
      ];
    }

    $form['pdf_file']['#states']['visible'][':input[name="export_external"]']['checked'] = TRUE;
    $form['submit'] = [
      '#type' => 'submit',
      '#value' => $this->t('Get Pdf File'),
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function validateForm(array &$form, FormStateInterface $form_state): void {
    parent::validateForm($form, $form_state);
    $export_external = $form_state->getValue('export_external');

    if ($export_external && !$this->file) {
      $all_files = $this->getRequest()->files->get('files', []);
      if (!empty($all_files['pdf_file'])) {
        try {
          $uploaded_files = $this->fileUploadHandler->handleFileUpload(
            $all_files['pdf_file'],
            $form_state->getValue('pdf_file'),
            $form['pdf_file']['#upload_validators'],
            'temporary://'
          );
          if ($uploaded_files && $uploaded_files instanceof FileInterface) {
            $this->file = $uploaded_files;
          }
        }
        catch (\Exception $e) {
          $form_state->setErrorByName('pdf_file', $this->t('PDF file upload failed: @error', ['@error' => $e->getMessage()]));
        }
      }
      if (!$this->file) {
        $form_state->setErrorByName('pdf_file', $this->t('PDF File not found.'));
      }
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    [$view_id, $display_id, $route_match] = $form_state->getBuildInfo()['args'];
    $args = $this->getViewArgs($route_match);

    $export_external = $form_state->getValue('export_external');
    if ($export_external) {
      $args['_pdf_file'] = $this->file;
    }
    $form_state->setResponse(PdfDataExport::buildResponse($view_id, $display_id, $args));
  }

  /**
   * Extracts view arguments from the route.
   *
   * @param \Drupal\Core\Routing\RouteMatchInterface|null $route_match
   *   The route match.
   *
   * @return array
   *   The view arguments.
   */
  protected function getViewArgs(?RouteMatchInterface $route_match = NULL): array {
    if (!isset($route_match)) {
      return [];
    }

    $route = $route_match->getRouteObject();
    $map = $route->getOption('_view_argument_map') ?? [];
    $args = [];

    foreach ($map as $attribute => $parameter_name) {
      $attribute = $map[$attribute] ?? $attribute;
      $arg = $route_match->getRawParameter($attribute) ?? $route_match->getParameter($attribute);

      if (isset($arg)) {
        $args[] = $arg;
      }
    }

    return $args;
  }

}
