<?php

namespace Drupal\views_data_export_tcpdf\Pdf;

/**
 * Custom TCPDF class with header and footer support.
 */
class CustomPdf extends \TCPDF {

  /**
   * Header configuration options.
   */
  protected array $headerConfig = [];

  /**
   * Footer configuration options.
   */
  protected array $footerConfig = [];

  /**
   * Sets the header configuration.
   */
  public function setHeaderConfig(array $config): void {
    $this->headerConfig = $config;
  }

  /**
   * Sets the footer configuration.
   */
  public function setFooterConfig(array $config): void {
    $this->footerConfig = $config;
  }

  /**
   * {@inheritdoc}
   */
  // phpcs:ignore Drupal.NamingConventions.ValidFunctionName.ScopeNotCamelCaps
  public function Header(): void {
    if (empty($this->headerConfig)) {
      return;
    }

    $header_title = $this->headerConfig['title'] ?? '';
    $header_lines = $this->headerConfig['lines'] ?? [];
    $logo_path = $this->headerConfig['logo_path'] ?? NULL;

    $this->SetFillColor(248, 249, 250);
    $this->Rect(0, 0, $this->getPageWidth(), 40, 'F');

    $top_margin = 10;
    $frame_left = PDF_MARGIN_LEFT;

    if ($logo_path) {
      $this->Image($logo_path, $frame_left, $top_margin, 25, 25, '', NULL, '', FALSE, 300, '', FALSE, FALSE, 0, FALSE, FALSE, FALSE);
      $frame_left += 30;
    }

    $this->SetXY($frame_left, $top_margin + 2);
    if ($header_title !== '') {
      $this->SetFont('helvetica', 'B', 16);
      $this->SetTextColor(15, 76, 129);
      $this->Cell(0, 7, $header_title, 0, 1, 'L');
    }

    if (!empty($header_lines)) {
      $this->SetFont('helvetica', '', 8);
      $this->SetTextColor(95, 108, 128);
      $this->SetX($frame_left);
      foreach ($header_lines as $line) {
        if ($line === '') {
          continue;
        }
        $this->Cell(0, 4, $line, 0, 1, 'L');
        $this->SetX($frame_left);
      }
    }

    $this->SetDrawColor(227, 232, 239);
    $this->Line(PDF_MARGIN_LEFT, 38, $this->getPageWidth() - PDF_MARGIN_RIGHT, 38, ['width' => 0.5]);
    $this->SetTextColor(0, 0, 0);
  }

  /**
   * {@inheritdoc}
   */
  // phpcs:ignore Drupal.NamingConventions.ValidFunctionName.ScopeNotCamelCaps
  public function Footer(): void {
    if (empty($this->footerConfig)) {
      return;
    }

    $footer_values = $this->footerConfig;

    $this->SetDrawColor(227, 232, 239);
    $this->Line(PDF_MARGIN_LEFT, $this->getPageHeight() - 20, $this->getPageWidth() - PDF_MARGIN_RIGHT, $this->getPageHeight() - 20, ['width' => 0.5]);

    $this->SetY(-16);
    $this->SetFont('helvetica', '', 7);
    $this->SetTextColor(95, 108, 128);

    $available_width = $this->getPageWidth() - PDF_MARGIN_LEFT - PDF_MARGIN_RIGHT;
    $segment_width = $available_width / 3;

    $this->setX(PDF_MARGIN_LEFT);
    $this->Cell($segment_width, 4, $footer_values['left'] ?? '', 0, 0, 'L');
    $this->Cell($segment_width, 4, $footer_values['center'] ?? '', 0, 0, 'C');
    $this->Cell($segment_width, 4, $footer_values['right'] ?? '', 0, 1, 'R');

    $this->SetY(-11);
    $this->SetX(PDF_MARGIN_LEFT);
    $this->SetFont('helvetica', 'I', 7);
    $this->Cell(0, 4, sprintf('Page %s of %s', $this->getAliasNumPage(), $this->getAliasNbPages()), 0, 0, 'R');

    $this->SetTextColor(0, 0, 0);
  }

}
