<?php

namespace Drupal\views_data_export_tcpdf\Plugin\views\style;

use Drupal\Component\Utility\Html;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\views_data_export\Plugin\views\style\DataExport;
use Symfony\Component\Serializer\SerializerInterface;

/**
 * A style plugin for Pdf export views.
 *
 * @ingroup views_style_plugins
 *
 * @ViewsStyle(
 *   id = "pdf_export",
 *   title = @Translation("Pdf export"),
 *   help = @Translation("Configurable row output for Pdf exports."),
 *   display_types = {"data"}
 * )
 */
class PdfDataExport extends DataExport {

  public function __construct(
    array $configuration,
    string $plugin_id,
    mixed $plugin_definition,
    SerializerInterface $serializer,
    array $serializer_formats,
    array $serializer_format_providers,
    ModuleHandlerInterface $module_handler,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $serializer, $serializer_formats, $serializer_format_providers, $module_handler);

    $this->formats = ['PDF'];
    $this->formatProviders = ['pdf' => 'pdf_data_export'];
  }

  /**
   * {@inheritdoc}
   */
  public function defineOptions(): array {
    $options = parent::defineOptions();
    $options['pdf_settings']['contains'] = [
      'pdf_format' => ['default' => 'PDF'],
      'header' => [
        'contains' => [
          'logo' => ['default' => NULL],
          'title' => ['default' => ''],
          'subtitle' => ['default' => ''],
          'phone' => ['default' => ''],
          'email' => ['default' => ''],
          'website' => ['default' => ''],
          'address' => ['default' => ''],
          'notes' => ['default' => ''],
        ],
      ],
      'footer' => [
        'contains' => [
          'left' => ['default' => ''],
          'center' => ['default' => ''],
          'right' => ['default' => ''],
        ],
      ],
    ];
    $options['pdf_settings']['contains']['metadata'] = [
      'contains' => [
        'creator' => ['default' => ''],
        'last_modified_by' => ['default' => ''],
        'title' => ['default' => ''],
        'description' => ['default' => ''],
        'subject' => ['default' => ''],
        'keywords' => ['default' => ''],
        'category' => ['default' => ''],
        'manager' => ['default' => ''],
        'company' => ['default' => ''],
      ],
    ];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);

    if ($form_state->get('section') === 'style_options') {
      $form['formats']['#type'] = 'radios';
      $form['formats']['#default_value'] = reset($this->options['formats']);
      unset($form['formats']['#description']);

      $pdf_options = $this->options['pdf_settings'];
      $form['pdf_settings'] = [
        '#type' => 'details',
        '#open' => TRUE,
        '#title' => $this->t('PDF settings'),
        '#tree' => TRUE,
      ];

      $header_defaults = $pdf_options['header'] ?? [];
      $form['pdf_settings']['header'] = [
        '#type' => 'details',
        '#title' => $this->t('Header content'),
        '#open' => TRUE,
        '#tree' => TRUE,
      ];

      $form['pdf_settings']['header']['logo'] = [
        '#type' => 'managed_file',
        '#title' => $this->t('Logo'),
        '#description' => $this->t('Upload a logo image to display in the PDF header.'),
        '#upload_location' => 'public://pdf-logos',
        '#upload_validators' => [
          'FileExtension' => [
            'extensions' => 'png jpg jpeg gif',
          ],
        ],
        '#default_value' => $header_defaults['logo'] ?? ($pdf_options['logo'] ?? NULL),
      ];

      $form['pdf_settings']['header']['title'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Header title'),
        '#description' => $this->t('Defaults to the view title when left empty.'),
        '#default_value' => $header_defaults['title'] ?? '',
      ];

      $form['pdf_settings']['header']['subtitle'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Subtitle or tagline'),
        '#default_value' => $header_defaults['subtitle'] ?? '',
      ];

      $form['pdf_settings']['header']['phone'] = [
        '#type' => 'textfield',
        '#title' => $this->t('Phone number'),
        '#default_value' => $header_defaults['phone'] ?? '',
      ];

      $form['pdf_settings']['header']['email'] = [
        '#type' => 'email',
        '#title' => $this->t('Email'),
        '#default_value' => $header_defaults['email'] ?? '',
      ];

      $form['pdf_settings']['header']['website'] = [
        '#type' => 'url',
        '#title' => $this->t('Website'),
        '#default_value' => $header_defaults['website'] ?? '',
      ];

      $form['pdf_settings']['header']['address'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Address'),
        '#rows' => 2,
        '#default_value' => $header_defaults['address'] ?? '',
      ];

      $form['pdf_settings']['header']['notes'] = [
        '#type' => 'textarea',
        '#title' => $this->t('Additional header notes'),
        '#rows' => 2,
        '#default_value' => $header_defaults['notes'] ?? '',
      ];

      $metadata = !empty($pdf_options['metadata']) ? array_filter($pdf_options['metadata']) : [];

      $form['pdf_settings']['metadata'] = [
        '#type' => 'details',
        '#title' => $this->t('Document metadata'),
        '#open' => !empty($metadata),
      ];

      $pdf_fields = [
        'creator' => $this->t('Author/creator name'),
        'last_modified_by' => $this->t('Last modified by'),
        'title' => $this->t('Title'),
        'description' => $this->t('Description'),
        'subject' => $this->t('Subject'),
        'keywords' => $this->t('Keywords'),
        'category' => $this->t('Category'),
        'manager' => $this->t('Manager'),
        'company' => $this->t('Company'),
      ];

      foreach ($pdf_fields as $pdf_field_key => $pdf_field_title) {
        $form['pdf_settings']['metadata'][$pdf_field_key] = [
          '#type' => 'textfield',
          '#title' => $pdf_field_title,
        ];

        if (isset($pdf_options['metadata'][$pdf_field_key])) {
          $form['pdf_settings']['metadata'][$pdf_field_key]['#default_value'] = $pdf_options['metadata'][$pdf_field_key];
        }
      }

      $footer_defaults = $pdf_options['footer'] ?? [];
      $form['pdf_settings']['footer'] = [
        '#type' => 'details',
        '#title' => $this->t('Footer content'),
        '#open' => !empty(array_filter($footer_defaults)),
        '#tree' => TRUE,
      ];

      $footer_fields = [
        'left' => $this->t('Left text'),
        'center' => $this->t('Center text'),
        'right' => $this->t('Right text'),
      ];

      foreach ($footer_fields as $footer_field_key => $footer_label) {
        $form['pdf_settings']['footer'][$footer_field_key] = [
          '#type' => 'textarea',
          '#title' => $footer_label,
          '#rows' => 2,
          '#description' => $this->t('Displayed in the @position area of every page footer.', ['@position' => $footer_field_key]),
          '#default_value' => $footer_defaults[$footer_field_key] ?? '',
        ];
      }
    }
  }

  /**
   * Attaches the PDF export link to the view.
   *
   * @param array $build
   *   The render array of the view.
   * @param string $display_id
   *   The display ID.
   * @param \Drupal\Core\Url $url
   *   The URL object for the export link.
   * @param string|null $title
   *   The title of the export link.
   */
  public function attachTo(array &$build, $display_id, Url $url, $title) {

    // @todo This mostly hard-codes PDF handling. Figure out how to abstract.
    $url_options = [];
    $input = $this->view->getExposedInput();
    if ($input) {
      $url_options['query'] = $input;
    }
    if ($pager = $this->view->getPager()) {
      $url_options['query']['page'] = $pager->getCurrentPage();
    }

    if (!empty($this->options['formats'])) {
      $url_options['query']['_format'] = reset($this->options['formats']);
    }
    $url->setOptions($url_options);
    $url = $url->toString();

    // Add the PDF icon to the view.
    $type = $this->displayHandler->getContentType();

    $this->view->feedIcons[] = [
      '#theme' => 'export_icon',
      '#url' => $url,
      '#type' => mb_strtoupper($type),
      '#theme_wrappers' => [
        'container' => [
          '#attributes' => [
            'class' => [
              Html::cleanCssIdentifier($type) . '-feed',
              'views-data-export-feed',
            ],
          ],
        ],
      ],
      '#attached' => [
        'library' => [
          'views_data_export_tcpdf/views_data_export_tcpdf',
        ],
      ],
    ];
    // Attach a link to the PDF feed, which is an alternate representation.
    $build['#attached']['html_head_link'][][] = [
      'rel' => 'alternate',
      'type' => $this->displayHandler->getMimeType(),
      'title' => $title,
      'href' => $url,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::submitOptionsForm($form, $form_state);

    // Make the uploaded logo file permanent.
    $logo_fid = $form_state->getValue(['pdf_settings', 'header', 'logo']);
    if (!empty($logo_fid)) {
      $fid = is_array($logo_fid) ? reset($logo_fid) : $logo_fid;
      if ($fid) {
        $file = \Drupal::entityTypeManager()->getStorage('file')->load($fid);
        if ($file) {
          $file->setPermanent();
          $file->save();
        }
      }
    }
  }

}
