<?php

declare(strict_types=1);

namespace Drupal\Tests\views_daterange_filters\Functional;

use Drupal\Core\Entity\EntityInterface;
use Drupal\field\Entity\FieldConfig;
use Drupal\field\Entity\FieldStorageConfig;
use Drupal\node\Entity\Node;
use Drupal\Tests\BrowserTestBase;
use Drupal\views\Views;

/**
 * Functional tests for views_daterange_filters operators.
 *
 * @group views_daterange_filters
 */
class ViewsDaterangeFiltersOperatorsTest extends BrowserTestBase {

  /**
   * {@inheritdoc}
   */
  protected static $modules = [
    'node',
    'user',
    'field',
    'text',
    'system',
    'datetime',
    'datetime_range',
    'optional_end_date',
    'views',
    'views_daterange_filters',
    // Test helper module that installs prebuilt Views via config.
    'views_daterange_filters_test',
  ];

  /**
   * {@inheritdoc}
   */
  protected $defaultTheme = 'stark';

  /**
   * Content type machine name.
   *
   * @var string
   */
  protected $bundle = 'event';

  /**
   * Daterange field name.
   *
   * @var string
   */
  protected $dateField = 'field_event_date';

  /**
   * Created node IDs keyed by label.
   *
   * @var array<string,int>
   */
  protected $nodes = [];

  /**
   * {@inheritdoc}
   */
  protected function setUp(): void {
    parent::setUp();

    // Create a content type with a daterange field.
    $this->drupalCreateContentType([
      'type' => $this->bundle,
      'name' => 'Event',
    ]);

    FieldStorageConfig::create([
      'field_name' => $this->dateField,
      'entity_type' => 'node',
      'type' => 'daterange',
      'cardinality' => 1,
      'settings' => [
        'datetime_type' => 'datetime',
        'optional_end_date' => TRUE,
      ],
      'translatable' => FALSE,
    ])->save();

    FieldConfig::create([
      'field_name' => $this->dateField,
      'entity_type' => 'node',
      'bundle' => $this->bundle,
      'label' => 'Event date',
      'required' => FALSE,
      'settings' => [
        'datetime_type' => 'datetime',
      ],
    ])->save();

    // Install optional_end_date module.
    optional_end_date_install();

    // Create some nodes with distinct ranges (UTC ISO format).
    $this->nodes['n1'] = $this->createEvent('Event 1', '2024-01-01T00:00:00', '2024-01-10T00:00:00');
    $this->nodes['n2'] = $this->createEvent('Event 2', '2024-01-05T00:00:00', '2024-01-05T23:59:59');
    $this->nodes['n3'] = $this->createEvent('Event 3', '2024-02-01T00:00:00', '2024-02-10T00:00:00');
    // Open-ended end.
    $this->nodes['n4'] = $this->createEvent('Event 4', '2024-01-20T00:00:00', NULL);
  }

  /**
   * Creates an event node.
   */
  protected function createEvent(string $title, ?string $start, ?string $end): EntityInterface {
    $values = [
      'type' => $this->bundle,
      'title' => $title,
      'status' => 1,
    ];
    $values[$this->dateField]['value'] = $start;
    if ($end !== NULL) {
      $values[$this->dateField]['end_value'] = $end;
    }
    $node = Node::create($values);
    $node->save();
    return $node;
  }

  /**
   * Helper to get node ID by key.
   *
   * @param string $key
   *   The node key.
   */
  protected function getNodeId(string $key): int {
    return (int) $this->nodes[$key]->id();
  }

  /**
   * Tests the "Includes" operator.
   */
  public function testIncludes(): void {
    $nids = $this->runDateViewFilter('includes', '2024-01-06T00:00:00');
    $this->assertEqualsCanonicalizing([
      $this->getNodeId('n1'),
      $this->getNodeId('n2'),
    ], $nids);
  }

  /**
   * Tests the "Includes (Unbound)" operator.
   */
  public function testIncludesUnbound(): void {
    $nids = $this->runDateViewFilter('includes_unbound', '2024-01-25T00:00:00');
    $this->assertContains($this->getNodeId('n4'), $nids);
    $this->assertNotContains($this->getNodeId('n1'), $nids);
    $this->assertNotContains($this->getNodeId('n2'), $nids);
    $this->assertNotContains($this->getNodeId('n3'), $nids);
  }

  /**
   * Tests the "Includes (Unbound Indexed)" operator.
   */
  public function testIncludesUnboundIndexed(): void {
    $nids = $this->runDateViewFilter('includes_unbound_indexed', '2024-01-25T00:00:00');
    $this->assertContains($this->getNodeId('n4'), $nids);
    $this->assertNotContains($this->getNodeId('n1'), $nids);
    $this->assertNotContains($this->getNodeId('n2'), $nids);
    $this->assertNotContains($this->getNodeId('n3'), $nids);
  }

  /**
   * Tests the "Overlaps" operator.
   */
  public function testOverlaps(): void {
    $nids = $this->runDateViewFilter('overlaps', [
      'min' => '2024-01-03T00:00:00',
      'max' => '2024-01-08T00:00:00',
    ]);
    $this->assertEqualsCanonicalizing([
      $this->getNodeId('n1'),
      $this->getNodeId('n2'),
    ], $nids);
  }

  /**
   * Tests the "Ends by" operator.
   */
  public function testEndsBy(): void {
    $nids = $this->runDateViewFilter('ends_by', '2024-01-12T00:00:00');
    $this->assertContains($this->getNodeId('n1'), $nids);
    $this->assertContains($this->getNodeId('n2'), $nids);
    $this->assertNotContains($this->getNodeId('n3'), $nids);
    $this->assertNotContains($this->getNodeId('n4'), $nids);
  }

  /**
   * Tests the "Not Ended" operator.
   *
   * The operator should return items that have not ended yet at the given
   * reference date (i.e., items with NULL end or end_value >= value).
   */
  public function testNotEnded(): void {
    // Use a date after n1 and n2 have ended, but while n3 and n4 have not.
    $nids = $this->runDateViewFilter('not_ended', '2024-01-25T00:00:00');
    // - n3 (2024-02-01..2024-02-10) has not ended by Jan 25 -> included.
    // - n4 (open-ended) -> included.
    // - n1 (ends 2024-01-10) -> excluded.
    // - n2 (ends 2024-01-05) -> excluded.
    $this->assertContains($this->getNodeId('n3'), $nids);
    $this->assertContains($this->getNodeId('n4'), $nids);
    $this->assertNotContains($this->getNodeId('n1'), $nids);
    $this->assertNotContains($this->getNodeId('n2'), $nids);
  }

  /**
   * Builds and executes a View with the daterange filter operator.
   *
   * @param string $operator
   *   The operator ID (includes, includes_unbound, includes_unbound_indexed,
   *   overlaps, ends_by).
   * @param array|string $value
   *   The filter value structure required by the operator.
   *
   * @return int[]
   *   The result node IDs.
   */
  protected function runDateViewFilter(string $operator, array|string $value): array {
    // Map operator to pre-installed view IDs from the test module config.
    $map = [
      'includes' => 'vdf_includes',
      'includes_unbound' => 'vdf_includes_unbound',
      'includes_unbound_indexed' => 'vdf_includes_unbound_indexed',
      'overlaps' => 'vdf_overlaps',
      'ends_by' => 'vdf_ends_by',
      'not_ended' => 'vdf_not_ended',
    ];
    $view_id = $map[$operator] ?? '';

    $executable = Views::getView($view_id);
    $this->assertNotNull($executable);
    // Pass along the exposed input expected by the filter operator. Since the
    // views in the test module expose the date filter, the input must be
    // keyed by the filter identifier (defaults to the filter ID).
    $executable->setDisplay('default');
    $executable->setExposedInput([
      'field_event_date_value' => $value,
    ]);
    $executable->executeDisplay('default');

    $nids = [];
    foreach ($executable->result as $row) {
      // Each row should expose nid from the base table.
      $nids[] = (int) $row->nid;
    }
    return $nids;
  }

}
