<?php

namespace Drupal\views_daterange_filters\Plugin\views\filter;

use Drupal\Component\Datetime\DateTimePlus;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\datetime\Plugin\Field\FieldType\DateTimeItemInterface;
use Drupal\datetime\Plugin\views\filter\Date;

/**
 * Date/time views filter.
 *
 * Extend Date filter to include date range operations.
 *
 * @ingroup views_filter_handlers
 *
 * @ViewsFilter("views_daterange_filters_daterange")
 */
class ViewsDaterangeFiltersDateRange extends Date implements ContainerFactoryPluginInterface {

  /**
   * {@inheritdoc}
   *
   * @return array
   *   Array of operators.
   */
  public function operators() {
    $operators = parent::operators();
    $operators['includes'] = [
      'title' => $this->t('Includes'),
      'method' => 'opIncludes',
      'short' => $this->t('includes'),
      'values' => 1,
    ];
    $operators['includes_unbound'] = [
      'title' => $this->t('Includes (Unbound)'),
      'method' => 'opIncludesUnbound',
      'short' => $this->t('includes unbound'),
      'values' => 1,
    ];
    $operators['includes_unbound_indexed'] = [
      'title' => $this->t('Includes (Unbound Indexed)'),
      'method' => 'opIncludesUnboundIndexed',
      'short' => $this->t('includes unbound indexed'),
      'values' => 1,
    ];
    $operators['overlaps'] = [
      'title' => $this->t('Overlaps'),
      'method' => 'opOverlaps',
      'short' => $this->t('within'),
      'values' => 2,
    ];
    $operators['ends_by'] = [
      'title' => $this->t('Ends by'),
      'method' => 'opEndsBy',
      'short' => $this->t('Ends by'),
      'values' => 1,
    ];
    $operators['not_ended'] = [
      'title' => $this->t('Not ended'),
      'method' => 'opNotEnded',
      'short' => $this->t('not ended'),
      'values' => 1,
    ];

    return $operators;
  }

  /**
   * Returns the matching end column name for a given start column name.
   */
  protected function getEndFieldName(string $field): string {
    return substr($field, 0, -6) . '_end_value';
  }

  /**
   * Formats a field column reference into a date expression suitable for SQL.
   */
  protected function formatField(string $field): string {
    return $this->query->getDateFormat(
      $this->query->getDateField($field, TRUE, $this->calculateOffset),
      $this->dateFormat,
      TRUE,
    );
  }

  /**
   * Converts an input value into a formatted date literal for SQL comparison.
   *
   * The value is interpreted in the active timezone, adjusted with the Views
   * origin offset, converted to storage (UTC), then wrapped as a Views date
   * expression.
   */
  protected function formatValue(string $input): string {
    $timezone = $this->getTimezone();
    $origin_offset = $this->getOffset($input, $timezone);
    $value = new DateTimePlus($input, new \DateTimeZone($timezone));
    return $this->query->getDateFormat(
      $this->query->getDateField(
        "'" . $this->dateFormatter->format(
          $value->getTimestamp() + $origin_offset,
          'custom',
          DateTimeItemInterface::DATETIME_STORAGE_FORMAT,
          DateTimeItemInterface::STORAGE_TIMEZONE,
        ) . "'",
        TRUE,
        $this->calculateOffset,
      ),
      $this->dateFormat,
      TRUE,
    );
  }

  /**
   * Converts an input value into a UTC string in the current date format.
   *
   * This variant is index-friendly for direct parameter binding.
   */
  protected function toUtcString(string $input): string {
    $timezone = $this->getTimezone();
    $origin_offset = $this->getOffset($input, $timezone);
    $value = new DateTimePlus($input, new \DateTimeZone($timezone));
    $timestamp = $value->getTimestamp() + $origin_offset;
    return gmdate($this->dateFormat, $timestamp);
  }

  /**
   * Filters by operator Includes.
   *
   * @param mixed $field
   *   The field.
   */
  protected function opIncludes($field) {
    $end_field = $this->getEndFieldName($field);

    $value = $this->formatValue($this->value['value']);
    $field = $this->formatField($field);
    $end_field = $this->formatField($end_field);

    $this->query->addWhereExpression($this->options['group'], "$value BETWEEN $field AND $end_field");
  }

  /**
   * Filters by operator IncludesUnbound.
   *
   * @param mixed $field
   *   The field.
   */
  protected function opIncludesUnbound($field) {
    $end_field = $this->getEndFieldName($field);

    $value_format = $this->formatValue($this->value['value']);
    $field_format = $this->formatField($field);
    $end_field_format = $this->formatField($end_field);

    $this->query->addWhereExpression(
      $this->options['group'],
      "(($field IS NULL OR $value_format >= $field_format)
          AND ($end_field IS NULL OR $value_format <= $end_field_format))",
    );
  }

  /**
   * Filters by operator IncludesUnboundIndexed.
   *
   * @param mixed $field
   *   The field.
   */
  protected function opIncludesUnboundIndexed($field) {
    $end_field = substr($field, 0, -6) . '_end_value';

    // Convert input to UTC storage format once.
    $value_utc = $this->toUtcString($this->value['value']);

    // Index-friendly SQL.
    $this->query->addWhereExpression(
      $this->options['group'],
      "(($field IS NULL OR $field <= :v) AND ($end_field IS NULL OR $end_field >= :v))",
      [':v' => $value_utc],
    );
  }

  /**
   * Filters by operator Overlaps.
   *
   * @param object $field
   *   The views field.
   */
  protected function opOverlaps($field) {
    $end_field = $this->getEndFieldName($field);

    // Although both 'min' and 'max' values are required, default empty 'min'
    // value as UNIX timestamp 0.
    $min = (!empty($this->value['min'])) ? $this->value['min'] : '@0';

    // Convert to ISO format and format for query. UTC timezone is used since
    // dates are stored in UTC.
    $a = $this->formatValue($min);
    $b = $this->formatValue($this->value['max']);

    $field = $this->formatField($field);
    $end_field = $this->formatField($end_field);

    $this->query->addWhereExpression($this->options['group'], "($a <= $end_field AND $b >= $field) OR ($a <= $field AND $b >= $field)");
  }

  /**
   * Filters by operator Ends By.
   *
   * @param mixed $field
   *   The field.
   */
  protected function opEndsBy($field) {
    $end_field = $this->getEndFieldName($field);

    $value = $this->formatValue($this->value['value']);
    $end_field = $this->formatField($end_field);

    $this->query->addWhereExpression($this->options['group'], "$end_field <= $value");
  }

  /**
   * Filters by operator Not ended.
   *
   * @param mixed $field
   *   The field.
   */
  protected function opNotEnded($field) {
    $end_field = $this->getEndFieldName($field);

    $value = $this->formatValue($this->value['value']);
    $end_field = $this->formatField($end_field);

    // Keep ranges that have not ended:
    // - end date is in the future
    // - OR end date is NULL (open-ended range)
    $this->query->addWhereExpression(
      $this->options['group'],
      "($end_field >= $value OR $end_field IS NULL)"
    );
  }

}
