<?php

namespace Drupal\views_entity_link\Plugin\views\field;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Routing\RedirectDestinationTrait;
use Drupal\views\Plugin\views\field\EntityLink as EntityLinkBase;
use Drupal\views\ResultRow;
use Symfony\Component\Serializer\Encoder\JsonDecode;
use Symfony\Component\Serializer\Encoder\JsonEncoder;

/**
 * Field handler to present a link to an entity using entity link templates.
 *
 * @ViewsField("views_entity_link")
 */
class EntityLink extends EntityLinkBase {

  use RedirectDestinationTrait;

  /**
   * {@inheritdoc}
   */
  public function defineOptions() {
    $options = parent::defineOptions();
    $options['link_template'] = ['default' => ''];
    $options['link_template_custom'] = ['default' => ''];
    $options['use_ajax'] = ['default' => FALSE];
    $options['dialog'] = ['default' => FALSE];
    $options['dialog_modal'] = ['default' => FALSE];
    $options['dialog_renderer'] = ['default' => ''];
    $options['dialog_options'] = ['default' => ''];
    $options['destination'] = ['default' => FALSE];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    $def = $this->entityTypeManager->getDefinition($this->definition['entity_type']);
    $templates = array_keys($def->getLinkTemplates());
    $templates = array_combine($templates, $templates);
    $templates['_custom'] = $this->t('Custom');

    $form['link_template'] = [
      '#type' => 'select',
      '#title' => $this->t('Link template'),
      '#description' => $this->t('Select link template.'),
      '#default_value' => $this->options['link_template'],
      '#options' => $templates,
      '#required' => TRUE,
    ];

    $form['link_template_custom'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Custom link template'),
      '#description' => $this->t('Use a custom link template.'),
      '#default_value' => $this->options['link_template_custom'],
      '#states' => [
        'visible' => [
          ':input[name="options[link_template]"]' => ['value' => '_custom'],
        ],
        'required' => [
          ':input[name="options[link_template]"]' => ['value' => '_custom'],
        ],
      ],
    ];

    $form['use_ajax'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Use AJAX'),
      '#description' => $this->t('Add the "use-ajax" class to the link.'),
      '#default_value' => $this->options['use_ajax'],
    ];

    $form['dialog'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Dialog'),
      '#description' => $this->t('Open the link in a dialog.'),
      '#default_value' => $this->options['dialog'],
    ];

    $form['dialog_modal'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Modal dialog'),
      '#description' => $this->t('Use a modal type dialog.'),
      '#default_value' => $this->options['dialog_modal'],
      '#states' => [
        'visible' => [
          ':input[name="options[dialog]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['dialog_renderer'] = [
      '#type' => 'select',
      '#title' => $this->t('Dialog renderer'),
      '#description' => $this->t('Select a dialog renderer. Note, not all renderers are compatible with all dialog/modal options.'),
      '#default_value' => $this->options['dialog_renderer'],
      '#options' => [
        'off_canvas' => $this->t('Off canvas'),
        'off_canvas_top' => $this->t('Off canvas top'),
      ],
      '#empty_option' => $this->t('- None -'),
      '#states' => [
        'visible' => [
          ':input[name="options[dialog]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['dialog_options'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Dialog options.'),
      '#description' => $this->t('Provide a JSON-encoded string of jQuery UI dialog options. Invalid JSON will be ignored.'),
      '#default_value' => $this->options['dialog_options'],
      '#states' => [
        'visible' => [
          ':input[name="options[dialog]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    $form['destination'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Include destination query parameter'),
      '#default_value' => $this->options['destination'],
    ];

    parent::buildOptionsForm($form, $form_state);

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function getEntityLinkTemplate() {
    return $this->options['link_template'];
  }

  /**
   * {@inheritdoc}
   */
  public function renderAsLink($alter, $text, $tokens) {
    $text = $this->viewsTokenReplace($text, $tokens);

    $render = [
      '#type' => 'link',
      '#title' => $text,
      '#url' => $alter['url'],
    ];

    if (!empty($this->options['use_ajax'])) {
      $render['#options']['attributes']['class'][] = 'use-ajax';
      $render['#attached']['library'][] = 'core/drupal.ajax';
      $render['#url']->setRouteParameter('js', 'nojs');
    }

    if ($this->options['dialog']) {
      $render['#options']['attributes']['class'][] = 'use-ajax';
      $render['#attached']['library'][] = 'core/drupal.dialog.ajax';
      $render['#options']['attributes']['data-dialog-type'] = $this->options['dialog_modal'] ? 'modal' : 'dialog';

      if ($this->options['dialog_renderer']) {
        $render['#options']['attributes']['data-dialog-renderer'] = $this->options['dialog_renderer'];
      }

      if ($this->options['dialog_options']) {
        try {
          $decoder = new JsonDecode();
          $decoded = $decoder->decode($this->options['dialog_options'], JsonEncoder::FORMAT);
        }
        catch (\Exception $e) {
          $decoded = FALSE;
        }
        if ($decoded) {
          $render['#options']['attributes']['data-dialog-options'] = $this->options['dialog_options'];
        }
      }
    }

    $this->getModuleHandler()->alter('views_entity_link', $render, $this->view, $this);
    \Drupal::theme()->alter('views_entity_link', $render, $this->view, $this);

    if (!empty($alter['prefix'])) {
      $render['#prefix'] = $this->viewsTokenReplace($alter['prefix'], $tokens);
    }
    if (!empty($alter['suffix'])) {
      $render['#suffix'] = $this->viewsTokenReplace($alter['suffix'], $tokens);
    }

    return $this->getRenderer()->render($render);
  }

  /**
   * {@inheritdoc}
   */
  protected function getUrlInfo(ResultRow $row) {
    $url = parent::getUrlInfo($row);

    if ($this->options['destination']) {
      $params = $url->getRouteParameters();
      $params['destination'] = $this->getRedirectDestination()->get();
      $url->setRouteParameters($params);
    }

    return $url;
  }

}
