<?php

namespace Drupal\views_field_view_exposed_filter\Plugin\views\filter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\filter\FilterPluginBase;
use Drupal\views\Views;

/**
 * Filters results strictly by comparing against an Exposed Source View.
 *
 * @ViewsFilter("exposed_source_filter")
 */
class ExposedSourceFilter extends FilterPluginBase {

  /**
   * {@inheritdoc}
   */
  public function canExpose() {
    return FALSE; // Disable "Expose" checkbox
  }

  /**
   * {@inheritdoc}
   */
  public function isExposed() {
    return FALSE; // Force internal use only
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['operator'] = ['default' => 'in'];
    $options['view_target'] = ['default' => ''];
    $options['view_display_target'] = ['default' => ''];
    $options['view_args'] = ['default' => ''];
    return $options;
  }

  /**
   * Define summary to prevent "Array to string" warnings.
   */
  public function adminSummary() {
    if (!empty($this->options['view_target'])) {
      $op = ($this->options['operator'] === 'not in') ? 'Exclude' : 'Include';
      return $this->t('@op Source View Results', ['@op' => $op]);
    }
    return $this->t('Not configured');
  }

  /**
   * Override validation to prevent "Element must be array" crash.
   */
  public function validateOptionsForm(&$form, FormStateInterface $form_state) {
     // Skip parent validation for exposed forms.
  }

  /**
   * Helper: Get only 'Exposed Source' displays.
   */
  protected function getDisplayOptions($view_id) {
    $display_options = [];
    if (!empty($view_id)) {
      $view_entity = Views::getView($view_id);
      if ($view_entity) {
        $view_entity->initDisplay();
        foreach ($view_entity->displayHandlers as $id => $display) {
          if ($display->getPluginId() === 'exposed_source') {
             $display_options[$id] = $display->display['display_title'];
          }
        }
      }
    }
    return $display_options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    $views = Views::getAllViews();
    $view_options = [];
    foreach ($views as $view) {
      if ($view->status()) {
        $displays = $view->get('display');
        foreach ($displays as $display) {
           if (isset($display['display_plugin']) && $display['display_plugin'] === 'exposed_source') {
             $view_options[$view->id()] = $view->label();
             break;
           }
        }
      }
    }

    $form['operator'] = [
      '#type' => 'radios',
      '#title' => $this->t('Operator'),
      '#options' => [
        'in' => $this->t('Is one of (Include rows found in Source View)'),
        'not in' => $this->t('Is not one of (Exclude rows found in Source View)'),
      ],
      '#default_value' => $this->options['operator'],
    ];

    $form['view_target'] = [
      '#type' => 'select',
      '#title' => $this->t('Source View'),
      '#options' => $view_options,
      '#default_value' => $this->options['view_target'],
      '#empty_option' => $this->t('- Select a View -'),
    ];

    $selected_view = $this->options['view_target'];

    if (!empty($selected_view)) {
      $display_options = $this->getDisplayOptions($selected_view);
      $form['view_display_target'] = [
        '#type' => 'select',
        '#title' => $this->t('Display'),
        '#options' => $display_options,
        '#default_value' => $this->options['view_display_target'],
        '#empty_option' => $this->t('- Select Display -'),
        '#validated' => TRUE,
      ];
    } else {
      $form['view_display_target'] = [
         '#markup' => '<div class="messages messages--warning">' . $this->t('Select a View above and click Apply.') . '</div>',
      ];
    }

    $form['view_args'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Arguments'),
      '#default_value' => $this->options['view_args'],
      '#description' => $this->t('Optional arguments for the source view.'),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $view_id = $this->options['view_target'];
    $display_id = $this->options['view_display_target'];
    $args_string = $this->options['view_args'];
    $operator = $this->options['operator'];

    if (empty($view_id) || empty($display_id)) {
      return;
    }

    $view = Views::getView($view_id);
    if (!$view) {
      return;
    }

    $view_args = !empty($args_string) ? array_map('trim', explode(',', $args_string)) : [];
    $view->setDisplay($display_id);
    $view->setArguments($view_args);

    $display_handler = $view->display_handler;
    $valid_ids = [];

    if ($display_handler->getPluginId() == 'exposed_source') {
      if (method_exists($display_handler, 'getExposedValues')) {
        $valid_ids = $display_handler->getExposedValues();
      }
    }

    $this->ensureMyTable();

    $op = ($operator === 'not in') ? 'NOT IN' : 'IN';

    if (empty($valid_ids)) {
      if ($op == 'IN') {
        $this->query->addWhereExpression($this->options['group'], '1 = 0');
      }
    }
    else {
      $this->query->addWhere($this->options['group'], "$this->tableAlias.$this->realField", $valid_ids, $op);
    }
  }
}
