<?php

namespace Drupal\views_field_view_exposed_filter\Plugin\views\style;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\style\StylePluginBase;

/**
 * Style plugin to extract a single field value.
 *
 * @ViewsStyle(
 * id = "views_field_view_exposed_source_style",
 * title = @Translation("Exposed Source Field"),
 * help = @Translation("Select the specific field to use as the filter value."),
 * theme = "views_view_unformatted",
 * display_types = {"exposed_source"}
 * )
 */
class ExposedSourceStyle extends StylePluginBase {

  /**
   * {@inheritdoc}
   */
  protected $usesRowPlugin = FALSE;

  /**
   * {@inheritdoc}
   */
  protected $usesFields = TRUE;

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['source_field'] = ['default' => ''];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    $fields = $this->displayHandler->getHandlers('field');
    $field_options = [];
    foreach ($fields as $field_id => $field) {
      $field_options[$field_id] = $field->adminLabel();
    }

    if (empty($field_options)) {
      $form['error_message'] = [
        '#markup' => '<div class="messages messages--warning">' . $this->t('Please add fields to this view first.') . '</div>',
      ];
      return;
    }

    $form['source_field'] = [
      '#type' => 'radios',
      '#title' => $this->t('Output Field (For Filter Logic)'),
      '#options' => $field_options,
      '#default_value' => $this->options['source_field'],
      '#description' => $this->t('Select the ONE field that contains the values needed for the filter (e.g. Entity ID).'),
      '#required' => TRUE,
    ];
  }

  /**
   * VISUALS: Renders a standard table for the Preview window.
   */
  public function render() {
    $this->renderFields($this->view->result);

    $rows = [];
    $header = [];

    foreach ($this->view->field as $field_id => $field) {
      if (empty($field->options['exclude'])) {
        $header[] = $field->label();
      }
    }

    foreach ($this->view->result as $index => $row) {
      $row_data = [];
      foreach ($this->view->field as $field_id => $field) {
        if (empty($field->options['exclude'])) {
          $row_data[] = $this->rendered_fields[$index][$field_id] ?? '';
        }
      }
      $rows[] = $row_data;
    }

    return [
      '#type' => 'table',
      '#header' => $header,
      '#rows' => $rows,
      '#empty' => $this->t('No results found.'),
      '#attributes' => ['class' => ['views-table', 'views-view-table']],
    ];
  }

  /**
   * LOGIC: Extracts only the specific field chosen in settings.
   */
  public function getSourceValues() {
    $source_field = $this->options['source_field'];
    $values = [];

    if (empty($this->view->result) || empty($source_field)) {
      return [];
    }

    foreach ($this->view->result as $row) {
      if (isset($this->view->field[$source_field])) {
        $val = $this->view->field[$source_field]->getValue($row);

        if (is_array($val)) {
             array_walk_recursive($val, function($v) use (&$values) {
                 $values[] = $v;
             });
        } elseif (!is_null($val) && $val !== '') {
            $values[] = $val;
        }
      }
    }

    return $values;
  }
}
