<?php

declare(strict_types=1);

namespace Drupal\views_insert_blocks\Form;

use Drupal\Core\Block\BlockManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormBuilderInterface;
use Drupal\Core\Form\FormState;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\Context\ContextRepositoryInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Displays a list of blocks to add to the view.
 */
final class BlockListForm extends ConfigFormBase {

  /**
   * Constructs a new BlockListForm object.
   */
  public function __construct(
    protected BlockManager $blockManager,
    protected ContextRepositoryInterface $contextRepository,
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typedConfigManager,
  ) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * Fields to ignore when saving config.
   */
  protected const IGNORE_FIELDS = [
    'admin_label',
  ];

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('plugin.manager.block'),
      $container->get('context.repository'),
      $container->get('config.factory'),
      $container->get('config.typed'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'views_insert_blocks_block_list';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['views_insert_blocks.settings'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    /** @var array $definitions */
    $definitions = $this->blockManager->getDefinitionsForContexts($this->contextRepository->getAvailableContexts());
    $blockOptions = [];

    foreach ($definitions as $block_id => $block_info) {
      $blockOptions[$block_id] = $block_info['admin_label'];
    }

    $form['block_list'] = [
      '#type' => 'container',
      '#prefix' => '<div id="block-list-wrapper">',
      '#suffix' => '</div>',
      '#tree' => TRUE,
    ];

    $form['block_list']['view_id'] = [
      '#type' => 'hidden',
      '#default_value' => $form_state->getValue('view_id'),
    ];

    $form['block_list']['display_id'] = [
      '#type' => 'hidden',
      '#default_value' => $form_state->getValue('display_id'),
    ];

    $form['block_list']['block_select'] = [
      '#title' => $this->t('Select a block.'),
      '#type' => 'select',
      '#description' => $this->t('Select a block to add to your view and configure it in next step.'),
      '#options' => $blockOptions,
      '#required' => TRUE,
      '#default_value' => $form_state->getValue('block_list')['block_select'] ?? array_key_first($blockOptions),
      '#ajax' => [
        'callback' => static::class . '::ajaxCallback',
        'wrapper' => 'block-list-wrapper',
        'event' => 'change',
        'options' => [
          'query' => [
            FormBuilderInterface::AJAX_FORM_REQUEST => TRUE,
          ],
        ],
      ],
    ];

    /** @var string $blockId */
    $blockId = $form_state->getValue('block_list')['block_select'] ?? array_key_first($blockOptions);
    /** @var \Drupal\Core\Block\BlockPluginInterface $pluginBlock */
    $pluginBlock = $this->blockManager->createInstance($blockId, []);
    $subformState = new FormState();
    /** @var array $blockForm */
    $blockForm = $pluginBlock->buildConfigurationForm([], $subformState);

    foreach ($blockForm as $key => $value) {

      if (isset($value["#title"]) && $value["#title"]->getUntranslatedString() === 'Title') {
        unset($value["#default_value"]);
      }

      if ($key !== 'label_display') {
        $form['block_list'][$key] = $value;
      }
    }

    $form['block_list']['block_position'] = [
      '#title' => $this->t('Select the position.'),
      '#type' => 'radios',
      '#description' => $this->t('Select the position on which you want this block to appear.'),
      '#options' => [
        'start' => $this->t('At the start of the view.'),
        'end' => $this->t('At the end of the view.'),
        'after' => $this->t('As n-th element of the view.'),
      ],
      '#required' => TRUE,
      '#default_value' => 'start',
    ];

    $form['block_list']['after'] = [
      '#description' => $this->t('Enter the element position for this block to appear at.'),
      '#title' => $this->t('Enter element position'),
      '#type' => 'number',
      '#min' => 1,
      '#max' => 2147483647,
      '#step' => 1,
      '#states' => [
        'visible' => [
          ':input[name="block_list[block_position]"]' => [
            'value' => 'after',
          ],
        ],
        'required' => [
          ':input[name="block_list[block_position]"]' => [
            'value' => 'after',
          ],
        ],
      ],
    ];

    $form['block_list']['block_page'] = [
      '#title' => $this->t('Select the page.'),
      '#type' => 'radios',
      '#description' => $this->t('Select the page on which you want this block to appear.'),
      '#options' => [
        'start' => $this->t('First page.'),
        'end' => $this->t('Last page.'),
        'after' => $this->t('N-th page of the view'),
        'nth' => $this->t('After every n-th page of the view.'),
        'all' => $this->t('On all the pages of the view.'),
      ],
      '#required' => TRUE,
      '#default_value' => 'start',
    ];

    $form['block_list']['page_after'] = [
      '#description' => $this->t('Enter the page number on which the block will appear.'),
      '#title' => $this->t('Enter a page'),
      '#type' => 'number',
      '#min' => 1,
      '#max' => 2147483647,
      '#step' => 1,
      '#states' => [
        'visible' => [
          ':input[name="block_list[block_page]"]' => [
            'value' => 'after',
          ],
        ],
        'required' => [
          ':input[name="block_list[block_page]"]' => [
            'value' => 'after',
          ],
        ],
      ],
    ];

    $form['block_list']['page_nth'] = [
      '#title' => $this->t('Enter a page'),
      '#description' => $this->t('Enter the number of pages in a set after which the block will appear.'),
      '#type' => 'number',
      '#min' => 1,
      '#max' => 2147483647,
      '#step' => 1,
      '#states' => [
        'visible' => [
          ':input[name="block_list[block_page]"]' => [
            'value' => 'nth',
          ],
        ],
        'required' => [
          ':input[name="block_list[block_page]"]' => [
            'value' => 'nth',
          ],
        ],
      ],
    ];

    return parent::buildForm($form, $form_state);
  }

  /**
   * Ajax callback to return the rebuilt form element.
   *
   * @param array &$form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   *
   * @return array
   *   Return our portion of the form.
   */
  public static function ajaxCallback(array &$form, FormStateInterface $form_state): array {
    return $form['block_list'];
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    /** @var array $values */
    $values = $form_state->getValue('block_list');
    $array = [];

    foreach ($values as $name => $value) {
      if (!in_array($name, self::IGNORE_FIELDS, TRUE)) {
        $array[$name] = $value;
      }
    }

    /** @var array $data */
    $data = $this->config('views_insert_blocks.settings')->getRawData();
    /** @var int $key */
    $key = count($data);

    if (NULL != $key) {
      $key = max(array_keys($data));
      $key += 1;
    }

    $this->config('views_insert_blocks.settings')
      ->set((string) $key, $array)
      ->save();

    $this->messenger()->addMessage($this->t('Block configuration saved.'));

    /** @var string $referer */
    $referer = $this->getRequest()->headers->get('referer');

    // Redirect user back to view configuration form after saving
    // the block config, this will also let our new config to load in the view.
    $response = new RedirectResponse($referer);
    $response->send();
  }

}
