<?php

declare(strict_types=1);

namespace Drupal\views_insert_blocks\Controller;

use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Form\FormState;

/**
 * Class ViewsInsertBlocksController.
 *
 * Returns forms to add blocks into the view or edit them.
 */
class ViewsInsertBlocksController extends ControllerBase {

  /**
   * Configuration name.
   */
  protected const CONFIG_NAME = 'views_insert_blocks.settings';

  /**
   * Dialog options.
   */
  protected const DIALOG_OPTIONS = [
    'minHeight' => 400,
    'minWidth' => 600,
    'resizable' => TRUE,
  ];

  /**
   * Name of the block list form.
   */
  protected const BLOCK_LIST_FORM = 'Drupal\views_insert_blocks\Form\BlockListForm';

  /**
   * Name of the block edit form.
   */
  protected const BLOCK_EDIT_FORM = 'Drupal\views_insert_blocks\Form\BlockEditForm';

  /**
   * Returns the block list config form in a modal ajax response.
   *
   * @return \Drupal\Core\Ajax\AjaxResponse
   *   Returns the ajax response.
   */
  public function getBlockList(string $view, string $display): AjaxResponse {
    $formState = new FormState();

    // Tell our form for which view and display we are saving the configuration.
    $formState->setValue('view_id', $view);
    $formState->setValue('display_id', $display);

    $form = $this->formBuilder()->buildForm(self::BLOCK_LIST_FORM, $formState);

    $response = new AjaxResponse();
    $response->addCommand(new OpenModalDialogCommand($this->t('Select a block to add into the view.'), $form, self::DIALOG_OPTIONS));

    return $response;
  }

  /**
   * Returns form that lets user edit or remove the added block.
   *
   * @param int $id
   *   The index of the block in the configuration.
   */
  public function getBlockConfiguration(int $id): AjaxResponse {
    $formState = new FormState();
    $config = $this->config(self::CONFIG_NAME);
    $data = $config->getRawData()[$id];

    // Let us populate form_state with default values for this block.
    foreach ($data as $key => $value) {
      $formState->setValue($key, $value);
    }

    $formState->setValue('block_index', $id);

    /** @var array $form */
    $form = $this->formBuilder()->buildForm(self::BLOCK_EDIT_FORM, $formState);

    $response = new AjaxResponse();
    $response->addCommand(new OpenModalDialogCommand($this->t('Edit block configuration.'), $form, self::DIALOG_OPTIONS));

    return $response;
  }

}
