<?php

declare(strict_types=1);

namespace Drupal\views_insert_blocks\Form;

use Drupal\Core\Block\BlockManager;
use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormState;
use Drupal\Core\Form\FormStateInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RedirectResponse;

/**
 * Displays an edit form for added block.
 */
final class BlockEditForm extends ConfigFormBase {

  /**
   * Constructs a new BlockEditForm object.
   */
    public function __construct(protected BlockManager $block_manager, ConfigFactoryInterface $config_factory, TypedConfigManagerInterface $typedConfigManager) {
    parent::__construct($config_factory, $typedConfigManager);
  }

  /**
   * Fields to ignore when saving config.
   */
  protected const IGNORE_FIELDS = [
    'block_configure',
    'admin_label',
    'block_index',
  ];

  /**
   * Form container types.
   */
  protected const CONTAINER_TYPES = [
    'fieldset',
    'vertical_tabs',
    'details',
    'container',
    'horizontal_tabs',
    'vertical_tab',
  ];

  /**
   * {@inheritdoc}
   */
  final public static function create(ContainerInterface $container): static {
    return new static(
      $container->get('plugin.manager.block'),
      $container->get('config.factory'),
      $container->get('config.typed'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId(): string {
    return 'views_insert_blocks_block_edit';
  }

  /**
   * {@inheritdoc}
   */
  protected function getEditableConfigNames(): array {
    return ['views_insert_blocks.block_edit'];
  }

  /**
   * {@inheritdoc}
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    if (NULL !== $form_state->getValue('block_select')) {
      /** @var array $defaultValues */
      $defaultValues = $form_state->getValues();
      /** @var string $blockId */
      $blockId = $form_state->getValue('block_select');
      /** @var \Drupal\Core\Block\BlockPluginInterface $pluginBlock */
      $pluginBlock = $this->block_manager->createInstance($blockId, []);
      $subformState = new FormState();
      /** @var array $blockForm */
      $blockForm = $pluginBlock->buildConfigurationForm([], $subformState);

      $form['block_list'] = [
        '#type' => 'container',
        '#prefix' => '<div id="block-edit-wrapper">',
        '#suffix' => '</div>',
        '#tree' => TRUE,
      ];

      $form['block_list']['block_select'] = [
        '#type' => 'hidden',
        '#default_value' => $form_state->getValue('block_select'),
      ];

      $form['block_list']['block_index'] = [
        '#type' => 'hidden',
        '#default_value' => $defaultValues['block_index'],
      ];

      $form['block_list']['view_id'] = [
        '#type' => 'hidden',
        '#default_value' => $form_state->getValue('view_id'),
      ];

      $form['block_list']['display_id'] = [
        '#type' => 'hidden',
        '#default_value' => $form_state->getValue('display_id'),
      ];

      foreach ($blockForm as $key => $value) {
        // If the form element is a container, look for fields inside it.
        if (array_key_exists('#type', $value) && in_array($value['#type'], self::CONTAINER_TYPES, TRUE)) {
          foreach ($value as $k => $v) {
            if (array_key_exists($k, $defaultValues)) {
              $value[$k]['#default_value'] = $defaultValues[$k];
            }
          }
        }
        else {
          if (array_key_exists($key, $defaultValues) && $key !== 'provider') {
            $value['#default_value'] = $defaultValues[$key];
          }
        }

        if ($key !== 'label_display') {
          $form['block_list'][$key] = $value;
        }
      }

      $form['block_list']['block_position'] = [
        '#title' => $this->t('Select the position.'),
        '#type' => 'radios',
        '#description' => $this->t('Select the position in which you want this block to appear.'),
        '#options' => [
          'start' => $this->t('At the start of the view.'),
          'end' => $this->t('At the end of the view.'),
          'after' => $this->t('After n-th element of the view.'),
        ],
        '#required' => TRUE,
        '#default_value' => $defaultValues['block_position'],
      ];

      $form['block_list']['after'] = [
        '#description' => $this->t('Enter the element position for this block to appear at.'),
        '#title' => $this->t('Enter element position'),
        '#type' => 'number',
        '#default_value' => $defaultValues['after'],
        '#min' => 1,
        '#step' => 1,
        '#states' => [
          'visible' => [
            ':input[name="block_list[block_position]"]' => [
              'value' => 'after',
            ],
          ],
          'required' => [
            ':input[name="block_list[block_position]"]' => [
              'value' => 'after',
            ],
          ],
        ],
      ];

      $form['block_list']['block_page'] = [
        '#title' => $this->t('Select the page.'),
        '#type' => 'radios',
        '#description' => $this->t('Select the page on which you want this block to appear.'),
        '#options' => [
          'start' => $this->t('First page.'),
          'end' => $this->t('Last page.'),
          'after' => $this->t('N-th page of the view'),
          'nth' => $this->t('After every n-th page of the view.'),
          'all' => $this->t('On all the pages of the view.'),
        ],
        '#required' => TRUE,
        '#default_value' => $defaultValues['block_page'],
      ];

      $form['block_list']['page_after'] = [
        '#description' => $this->t('Enter the page number on which the block will appear.'),
        '#title' => $this->t('Enter a page'),
        '#type' => 'number',
        '#default_value' => $defaultValues['page_after'],
        '#min' => 1,
        '#step' => 1,
        '#states' => [
          'visible' => [
            ':input[name="block_list[block_page]"]' => [
              'value' => 'after',
            ],
          ],
          'required' => [
            ':input[name="block_list[block_page]"]' => [
              'value' => 'after',
            ],
          ],
        ],
      ];

      $form['block_list']['page_nth'] = [
        '#title' => $this->t('Enter a page'),
        '#description' => $this->t('Enter the number of pages in a set after which the block will appear.'),
        '#type' => 'number',
        '#default_value' => $defaultValues['page_nth'],
        '#min' => 1,
        '#step' => 1,
        '#states' => [
          'visible' => [
            ':input[name="block_list[block_page]"]' => [
              'value' => 'nth',
            ],
          ],
          'required' => [
            ':input[name="block_list[block_page]"]' => [
              'value' => 'nth',
            ],
          ],
        ],
      ];

      $form['actions']['remove'] = [
        '#type' => 'submit',
        '#value' => $this->t('Remove block'),
        '#name' => 'remove',
        '#submit' => [static::class . '::removeBlockCallback'],
      ];
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Ajax callback to set a block to configure.
   *
   * @param array &$form
   *   The form array.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   Form state object.
   */
  public static function removeBlockCallback(array &$form, FormStateInterface $form_state): void {
    $indexToRemove = (string) $form_state->getValues()['block_list']['block_index'];

    $config = \Drupal::service('config.factory')->getEditable('views_insert_blocks.settings');
    $config = $config->clear($indexToRemove);
    $config->save();

    \Drupal::messenger()->addMessage(t('Block successfully removed.'));

    /** @var string $referer */
    $referer = \Drupal::request()->headers->get('referer');

    $response = new RedirectResponse($referer);
    $response->send();
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    /** @var array $values */
    $values = $form_state->getValue('block_list');
    $array = [];

    foreach ($values as $name => $value) {
      if (!in_array($name, self::IGNORE_FIELDS, TRUE)) {
        $array[$name] = $value;
      }
    }

    $index = (string) $values['block_index'];

    $this->configFactory()->getEditable('views_insert_blocks.settings')
      ->set($index, $array)
      ->save();

    $this->messenger()->addMessage($this->t('Block configuration saved.'));

    /** @var string $referer */
    $referer = $this->getRequest()->headers->get('referer');

    // Redirect user back to view configuration form after saving
    // the block config, this will also let our new config to load in the view.
    $response = new RedirectResponse($referer);
    $response->send();
  }

}
