<?php

namespace Drupal\views_lazy_renderer\Plugin\views\row;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\search_api\IndexInterface;
use Drupal\search_api\Plugin\views\query\SearchApiQuery;
use Drupal\search_api\SearchApiException;
use Drupal\search_api\Utility\Utility;
use Drupal\views\Attribute\ViewsRow;
use Drupal\views\Plugin\views\display\DisplayPluginBase;
use Drupal\views\Plugin\views\row\RowPluginBase;
use Drupal\views\ViewExecutable;

#[ViewsRow(
  id: "lazy_row",
  title: new TranslatableMarkup("Lazy row"),
  theme: "views_view_fields",
  display_types: ["normal"]
)]
final class LazySearchRow extends RowPluginBase {

  protected ?IndexInterface $index = NULL;

  public function defineOptions(): array {
    $options = parent::defineOptions();
    $options['view_modes'] = ['default' => []];
    return $options;
  }

  public function init(ViewExecutable $view, DisplayPluginBase $display, array &$options = NULL) {
    parent::init($view, $display, $options);

    $base_table = $view->storage->get('base_table');
    try {
      $this->index = SearchApiQuery::getIndexFromTable($base_table, \Drupal::entityTypeManager());
    }
    catch (\Exception $e) {
      // Disable plugin for non-Search API Views and log an error.
      \Drupal::messenger()->addError($this->t(
        'The "Lazy row" plugin can only be used with Search API Views.'
      ));
      $this->index = NULL;
    }
  }

  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    if (!$this->index) {
      return;
    }

    parent::buildOptionsForm($form, $form_state);

    foreach ($this->index->getDatasources() as $datasource_id => $datasource) {
      $view_modes = $datasource->getViewModes();

      $title = $this->t('View mode for datasource %datasource', ['%datasource' => $datasource->label()]);

      $form['view_modes'][$datasource_id] = ['#type' => 'container'];
      $form['view_modes'][$datasource_id]['view_mode'] = [
        '#type' => 'select',
        '#title' => $title,
        '#options' => $view_modes,
        '#default_value' => $this->options['view_modes'][$datasource_id]['view_mode'] ?? key($view_modes),
      ];

      // Alternating view modes.
      $alternating = $this->options['view_modes'][$datasource_id]['alternating_fieldset']['alternating'] ?? FALSE;
      $all_pages = $this->options['view_modes'][$datasource_id]['alternating_fieldset']['allpages'] ?? FALSE;

      $form['view_modes'][$datasource_id]['alternating_fieldset'] = [
        '#type' => 'details',
        '#title' => $this->t('Alternating view mode'),
        '#open' => $alternating,
      ];
      $form['view_modes'][$datasource_id]['alternating_fieldset']['alternating'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Use the changing view mode selector'),
        '#default_value' => $alternating,
      ];
      $form['view_modes'][$datasource_id]['alternating_fieldset']['allpages'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Apply this configuration on every page'),
        '#default_value' => $all_pages,
      ];

      // Per-item view modes.
      $limit = $this->view->display_handler->getPlugin('pager')->getItemsPerPage();
      if ($limit > 0 && $limit <= 20) {
        for ($i = 0; $i < $limit; $i++) {
          $form['view_modes'][$datasource_id]['alternating_fieldset']['items']["item_$i"]['view_mode'] = [
            '#type' => 'select',
            '#title' => $this->t('Item @nr', ['@nr' => $i + 1]),
            '#options' => $view_modes,
            '#default_value' => $this->options['view_modes'][$datasource_id]['alternating_fieldset']['items']["item_$i"]['view_mode'] ?? 'teaser',
            '#states' => [
              'visible' => [
                ':input[name="row_options[view_modes][' . $datasource_id . '][alternating_fieldset][alternating]"]' => ['checked' => TRUE],
              ],
            ],
          ];
        }
      }
      else {
        $form['view_modes'][$datasource_id]['alternating_fieldset']['disabled_notice'] = [
          '#markup' => $this->t('Alternating view mode is disabled for unlimited or large item listings.'),
        ];
      }
    }
  }

  public function render($row): array {
    if (!$this->index) {
      return [];
    }

    [$datasource_id, $raw_id] = Utility::splitCombinedId($row->_item->getId());
    [$entity_id, ] = Utility::splitPropertyPath($raw_id);
    $datasource = $this->index->getDatasource($datasource_id);
    $bundles = array_keys($datasource->getBundles());
    $bundle = reset($bundles);
    $entity_type_id = $datasource->getEntityTypeId();

    if (!in_array($entity_type_id, ['node', 'media'], TRUE)) {
      return [];
    }

    $view_mode = $this->options['view_modes'][$datasource_id]['view_mode'] ?? 'default';
    $alternating_enabled = !empty($this->options['view_modes'][$datasource_id]['alternating_fieldset']['alternating']);
    $apply_all_pages = !empty($this->options['view_modes'][$datasource_id]['alternating_fieldset']['allpages']);

    if ($alternating_enabled) {
      $page = $this->view->getPager()->getCurrentPage();
      if ($page > 0 && !$apply_all_pages) {
        $view_mode = $this->options['view_modes'][$datasource_id]['view_mode'] ?? 'default';
      }
      else {
        $item_key = "item_{$row->index}";
        $view_mode = $this->options['view_modes'][$datasource_id]['alternating_fieldset']['items'][$item_key]['view_mode'] ?? $view_mode;
      }
    }

    try {
      $build = [
        '#theme' => 'views_lazy_row',
        '#bundle' => $bundle,
        '#parameters' => [
          'view_mode' => $view_mode,
          $entity_type_id => $entity_id,
        ],
        '#entity_type_id' => $entity_type_id,
      ];

      // Add the excerpt to the render array to allow adding it to view modes.
      if (isset($row->search_api_excerpt)) {
        $build['#search_api_excerpt'] = $row->search_api_excerpt;
      }

      if (!in_array('views_lazy_renderer/htmx', $this->view->element['#attached']['library'] ?? [], TRUE)) {
        $this->view->element['#attached']['library'][] = 'views_lazy_renderer/htmx'; // TODO: enable
      }

      return $build;
    }
    catch (SearchApiException $e) {
      \Drupal::logger('views_lazy_renderer')->error(
        'Lazy row failed for entity @id: @message',
        ['@id' => $entity_id, '@message' => $e->getMessage()]
      );

      return [];
    }
  }

}
