<?php

namespace Drupal\views_lazy_renderer\Controller;

use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Cache\CacheableResponse;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Entity\EntityInterface;
use Drupal\Core\Render\RendererInterface;
use Drupal\media\MediaInterface;
use Drupal\node\NodeInterface;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpKernel\Exception\AccessDeniedHttpException;

class PartialRenderController extends ControllerBase {

  public function __construct(
    private RendererInterface $renderer,
  ) {}

  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer'),
    );
  }

  public function node(NodeInterface $node, string $view_mode): Response {
    return $this->renderPartial($node, $view_mode);
  }

  public function media(MediaInterface $media, string $view_mode): Response {
    return $this->renderPartial($media, $view_mode);
  }

  private function renderPartial(EntityInterface $entity, string $view_mode): Response {
    $allowed = $this->viewModeAllowed($entity, $view_mode);
    if (!$allowed) {
      throw new AccessDeniedHttpException("Lazy rendering is not allowed for this bundle or view mode.");
    }

    $viewbuilder = \Drupal::entityTypeManager()->getViewBuilder($entity->getEntityTypeId());
    $build = $viewbuilder->view($entity, $view_mode);

    $response = new CacheableResponse($this->renderer->renderRoot($build));
    $response->getCacheableMetadata()
      ->addCacheContexts(['url.query_args:view_mode_name'])
      ->addCacheableDependency($entity)
      ->addCacheableDependency(CacheableMetadata::createFromRenderArray($build));

    return $response;
  }

  private function viewModeAllowed(EntityInterface $entity, string $view_mode): bool {
    $bundle = $entity->bundle();
    $entity_type_id = $entity->getEntityTypeId();

    $bundle_storage = \Drupal::entityTypeManager()->getStorage("$entity_type_id" . "_type");
    $bundle_config = $bundle_storage->load($bundle);
    if (!$bundle_config) {
      return FALSE;
    }

    // Check if lazy rendering is allowed for this bundle.
    $allowed = $bundle_config->getThirdPartySetting('views_lazy_renderer', 'allow_lazy_rendering', FALSE);
    $allowed_view_modes = $bundle_config->getThirdPartySetting('views_lazy_renderer', 'allow_lazy_rendering_view_modes', []);

    return $allowed && in_array($view_mode, $allowed_view_modes, TRUE);
  }

}
