<?php

namespace Drupal\views_menu_children_filter\Plugin\views\filter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsFilter;
use Drupal\views\Plugin\views\filter\FilterPluginBase;
use Drupal\views_menu_children_filter\Plugin\views\join\MenuChildrenNodeJoin;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Filter handler for menu item enabled status.
 *
 * @ingroup views_filter_handlers
 */
#[ViewsFilter("menu_children_enabled")]
class MenuItemEnabledFilter extends FilterPluginBase {

  /**
   * A join handler object.
   *
   * @var \Drupal\views_menu_children_filter\Plugin\views\join\MenuChildrenNodeJoin
   */
  protected $joinHandler;

  /**
   * MenuChildren constructor.
   *
   * @param \Drupal\views_menu_children_filter\Plugin\views\join\MenuChildrenNodeJoin $join_handler
   *   A join handler object.
   * @param array $configuration
   *   A configuration array.
   * @param string $plugin_id
   *   A plugin id.
   * @param mixed $plugin_definition
   *   A plugin definition.
   */
  public function __construct(MenuChildrenNodeJoin $join_handler, array $configuration, $plugin_id, $plugin_definition) {
    $this->joinHandler = $join_handler;
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * Creates an instance of the plugin.
   *
   * @param \Symfony\Component\DependencyInjection\ContainerInterface $container
   *   The container to pull out services used in the plugin.
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   *
   * @return static
   *   Returns an instance of this plugin.
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $container->get('views_menu_children_filter.join_handler'),
      $configuration,
      $plugin_id,
      $plugin_definition
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();
    $options['value'] = ['default' => 1];
    return $options;
  }

  /**
   * {@inheritdoc}
   */
  protected function valueForm(&$form, FormStateInterface $form_state) {
    $form['value'] = [
      '#type' => 'radios',
      '#title' => $this->t('Menu item status'),
      '#options' => [
        1 => $this->t('Enabled'),
        0 => $this->t('Disabled'),
      ],
      '#default_value' => $this->value ?? 1,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $this->joinHandler->joinToNodeTable($this->query);

    // Add the WHERE condition.
    $status = !empty($this->value) ? 1 : 0;
    $this->query->addWhere(
      $this->options['group'],
      "$this->tableAlias.enabled",
      $status,
      '='
    );
  }

  /**
   * {@inheritdoc}
   */
  public function adminSummary() {
    return $this->t('Menu item: @status', [
      '@status' => !empty($this->value) ? $this->t('Enabled') : $this->t('Disabled'),
    ]);
  }

}
