<?php

namespace Drupal\views_mobile\Hook;

use Drupal\Core\Hook\Attribute\Hook;
use Drupal\Core\Routing\RouteMatchInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\views\ViewExecutable;
use Drupal\views_mobile\Service\DisplaySwitcher;
use Drupal\Core\Logger\LoggerChannelFactoryInterface;
use Drupal\Core\Logger\LoggerChannelInterface;

/**
 * Hook implementations for views.
 */
class ViewsMobileHooks {

  use StringTranslationTrait;

  /**
   * The display switcher service.
   *
   * @var \Drupal\views_mobile\Service\DisplaySwitcher
   */
  protected DisplaySwitcher $switcher;

  /**
   * Logger channel for the module.
   *
   * @var \Drupal\Core\Logger\LoggerChannelInterface
   */
  protected LoggerChannelInterface $logger;

  /**
   * Constructor: dependencies are autowired when the attribute class is
   * registered as a service.
   */
  public function __construct(LoggerChannelFactoryInterface $logger_factory, DisplaySwitcher $switcher) {
    $this->logger = $logger_factory->get('views_mobile');
    $this->switcher = $switcher;
  }

  /**
   * Implements hook_help().
   */
  #[Hook('help')]
  public function help($route_name, RouteMatchInterface $route_match): ?string  {
    switch ($route_name) {
      case 'help.page.views_mobile':
        return $this->t('Enable to switch between desktop and mobile display version of your views.');
    }

    return NULL;
  }


  /**
   * Implements hook_views_pre_view().
   */
  #[Hook('views_pre_view')]
  /**
   * Implementation of hook_views_pre_view().
   *
   * Delegates to the views_mobile.display_switcher service, so the logic can be
   * dependency-injected and unit-tested.
   */
  public function pre_view(ViewExecutable $view, $display_id, array &$args) {
    try {
      $this->switcher->switchDisplay($view, $display_id, $args);
    }
    catch (\Throwable $e) {
      // Fail gracefully and allow the view to render as-is. Log the exception
      // for debugging.
      $this->logger->error($e->getMessage());
    }
  }
}
