<?php

declare(strict_types=1);

namespace Drupal\tests\views_mobile\Kernel;

use Drupal\KernelTests\KernelTestBase;
use Detection\MobileDetect;
use Drupal\views\ViewExecutable;

/**
 * Kernel tests for the Views Mobile integration.
 *
 * This test verifies that attribute-based hook implementations are invoked and
 * that the DisplaySwitcher service is used to switch view displays when the
 * mobile_detect service indicates a mobile device.
 */
class ViewsMobileKernelTest extends KernelTestBase {

  /**
   * Modules to enable.
   *
   * @var array
   */
  protected static $modules = [
    'system',
    'views',
    'mobile_detect',
    'views_mobile',
  ];

  protected function setUp(): void {
    parent::setUp();

    // Replace the mobile_detect service with a test double that reports
    // isMobile() = TRUE and isTablet() = FALSE.
    $mock = $this->createMock(MobileDetect::class);
    $mock->method('isMobile')->willReturn(TRUE);
    $mock->method('isTablet')->willReturn(FALSE);

    // The container in KernelTestBase is mutable for test purposes.
    $this->container->set('mobile_detect', $mock);
  }

  public function testViewsPreViewHookUsesDisplaySwitcher(): void {
    // Create a lightweight stub of ViewExecutable. We'll inject a simple
    // displayHandlers object with a has() method and a mock for setDisplay().
    $view = $this->getMockBuilder(ViewExecutable::class)
      ->disableOriginalConstructor()
      ->getMock();

    // Create a stub displayHandlers object that reports having 'default_mobile'.
    $displayHandlers = new class {
      protected $ids = ['default_mobile' => 'default_mobile'];
      public function has($id) { return isset($this->ids[$id]); }
    };

    // Inject the displayHandlers property into the view mock.
    $ref = new \ReflectionProperty($view, 'displayHandlers');
    $ref->setAccessible(true);
    $ref->setValue($view, $displayHandlers);

    // Expect that setDisplay() is called with the mobile display id.
    $view->expects($this->once())->method('setDisplay')->with('default_mobile');

    $args = [];

    // Invoke the hook implementations via the module handler. This will call
    // into the attribute-based hook class which should delegate to the
    // DisplaySwitcher service, which will then call setDisplay on our view.
    \Drupal::moduleHandler()->invokeAll('views_pre_view', [$view, 'default', &$args]);
  }

}

