<?php

declare(strict_types=1);

namespace Drupal\views_mobile\Service;

use Drupal\views\ViewExecutable;
use Detection\MobileDetect;
use Psr\Log\LoggerInterface;

/**
 * Service that switches View displays for mobile/tablet devices.
 */
class DisplaySwitcher {

  /**
   * The mobile detect service.
   *
   * @var \Detection\MobileDetect|null
   */
  protected ?MobileDetect $mobileDetect;

  /**
   * Logger.
   *
   * @var \Psr\Log\LoggerInterface
   */
  protected LoggerInterface $logger;

  /**
   * Constructor.
   */
  public function __construct(LoggerInterface $logger, ?MobileDetect $mobile_detect = NULL, ) {
    $this->logger = $logger;
    $this->mobileDetect = $mobile_detect;
  }

  /**
   * Switch a view display based on device detection.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view executable.
   * @param string $display_id
   *   The current display id.
   * @param array $args
   *   View arguments (passed by reference in hook but here not modified).
   */
  public function switchDisplay(ViewExecutable $view, string $display_id, array &$args): void {
    // If no detector available, try to instantiate as a fallback.
    $mobileDetector = $this->mobileDetect;
    if ($mobileDetector === NULL && class_exists(MobileDetect::class)) {
      $mobileDetector = new MobileDetect();
    }

    if ($mobileDetector === NULL) {
      // Nothing we can do.
      return;
    }

    // Ensure display handlers are initialized.
    if (method_exists($view, 'initDisplay')) {
      $view->initDisplay();
    }

    $display_id_mobile = $display_id . '_mobile';
    $display_id_tablet = $display_id . '_tablet';

    $has_mobile_mobile = isset($view->displayHandlers) && $view->displayHandlers->has($display_id_mobile);
    $has_mobile_tablet = isset($view->displayHandlers) && $view->displayHandlers->has($display_id_tablet);

    try {
      if (method_exists($mobileDetector, 'isTablet') && $mobileDetector->isTablet() && $has_mobile_tablet) {
        $view->setDisplay($display_id_tablet);
      }
      elseif (method_exists($mobileDetector, 'isMobile') && $mobileDetector->isMobile() && $has_mobile_mobile) {
        $view->setDisplay($display_id_mobile);
      }
    }
    catch (\Throwable $e) {
      // Log and rethrow? We choose to log and swallow to avoid breaking page.
      $this->logger->error('views_mobile: device detection error: @message', ['@message' => $e->getMessage()]);
    }
  }

}

