<?php

namespace Drupal\views_published_or_roles\Plugin\views\filter;

use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Session\AccountInterface;
use Drupal\user\Entity\Role;
use Drupal\user\RoleInterface;
use Drupal\views\Plugin\views\filter\FilterPluginBase;

/**
 * Filter by published status and by role.
 *
 * @ingroup views_filter_handlers
 *
 * @ViewsFilter("status_has_role")
 */
class PublishedOrHasRoles extends FilterPluginBase {

  /**
   * {@inheritdoc}
   */
  public function canExpose() {
    return FALSE;
  }

  /**
   * Display the filter on the administrative summary.
   */
  public function adminSummary() {
    $roles = NULL;
    if (is_array($this->value)) {
      $roles = array_keys($this->value);
      $roles = implode(' ', $roles);
    }
    return $this->operator . ' ' . $roles;
  }

  /**
   * Returns available role options (custom roles only).
   *
   * @return array
   *   Array keyed by role ID with human-readable labels.
   */
  private function getValueOptions(): array {
    if (function_exists('user_role_names')) {
      // Drupal ≤10.x: TRUE excludes anonymous; we then remove authenticated.
      $options = user_role_names(TRUE);
    }
    else {
      // Drupal 11+: replicate user_role_names(TRUE) behavior.
      // Load all roles keyed by role ID.
      $roles = Role::loadMultiple();
      // Exclude anonymous (what the TRUE flag did).
      unset($roles[RoleInterface::ANONYMOUS_ID]);

      // Build id => label while preserving keys.
      $ids = array_keys($roles);
      $labels = array_map(static fn(RoleInterface $r) => $r->label(), $roles);
      $options = array_combine($ids, $labels) ?: [];
    }

    // Match your original code: remove authenticated; keep only custom roles.
    unset($options[AccountInterface::AUTHENTICATED_ROLE]);

    // Optional: stable, human-friendly order.
    asort($options, SORT_NATURAL | SORT_FLAG_CASE);

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  protected function valueForm(&$form, FormStateInterface $form_state) {
    $form['value'] = [
      '#type' => 'select',
      '#title' => t('Select Role(s)'),
      '#size' => 30,
      '#options' => $this->getValueOptions(),
      '#default_value' => $this->value,
      '#multiple' => TRUE,
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function query() {
    $table = 'node_field_data';
    $roles = [];
    if (is_array($this->value)) {
      $roles = array_keys($this->value);
    }
    if (empty($roles)) {
      // If there are no roles selected, not querying on roles.
      $this->query->addWhereExpression(
        $this->options['group'],
        "$table.status = 1
        OR
        ($table.uid = ***CURRENT_USER*** AND ***CURRENT_USER*** <> 0 AND ***VIEW_OWN_UNPUBLISHED_NODES*** = 1)
        OR
        ***BYPASS_NODE_ACCESS*** = 1");
    }
    else {
      $this->query->addWhereExpression(
        $this->options['group'],
        "$table.status = 1
        OR
        ($table.uid = ***CURRENT_USER*** AND ***CURRENT_USER*** <> 0 AND ***VIEW_OWN_UNPUBLISHED_NODES*** = 1)
        OR
        ***BYPASS_NODE_ACCESS*** = 1
        OR
        ***CURRENT_USER*** IN (SELECT ur.entity_id FROM {user__roles} ur WHERE ur.roles_target_id IN (:roles[]))", [':roles[]' => $roles]);
    }
  }

  /**
   * Skip validation if no options have been chosen.
   */
  public function validate() {
    if (!empty($this->value)) {
      parent::validate();
    }
  }

}
