<?php

declare(strict_types=1);

namespace Drupal\views_query\Plugin\views\query;

use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\views\Plugin\views\query\DateSqlInterface;
use Drupal\views\Plugin\views\query\Sql as SqlCore;
use Drupal\views_query\ViewsQueryPluginManager;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Views query plugin that uses views_query plugins.
 */
class ViewsQuerySql extends SqlCore {

  /**
   * The cached views_query instances.
   *
   * @var \Drupal\views_query\ViewsQueryInterface[]
   */
  protected array $viewsQueryInstances = [];

  /**
   * Constructs a ViewsQuerySql object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entity_type_manager
   *   The entity type manager.
   * @param \Drupal\views\Plugin\views\query\DateSqlInterface $date_sql
   *   The database-specific date handler.
   * @param \Drupal\Core\Messenger\MessengerInterface $messenger
   *   The messenger.
   * @param \Drupal\views_query\ViewsQueryPluginManager $viewsQueryManager
   *   The views_query plugin manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    EntityTypeManagerInterface $entity_type_manager,
    DateSqlInterface $date_sql,
    MessengerInterface $messenger,
    protected readonly ViewsQueryPluginManager $viewsQueryManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition, $entity_type_manager, $date_sql, $messenger);

    $definitions = $this->viewsQueryManager->getDefinitions();
    foreach (array_keys($definitions) as $id) {
      $this->viewsQueryInstances[$id] = $this->viewsQueryManager->createInstance($id);
    }
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
      $container->get('views.date_sql'),
      $container->get('messenger'),
      $container->get('plugin.manager.views_query'),
    );
  }

  /**
   * {@inheritdoc}
   */
  protected function defineOptions(): array {
    $options = parent::defineOptions();

    foreach ($this->viewsQueryInstances as $instance) {
      $options = array_merge($options, $instance->defineOptions());
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   *
   * @param array $form
   *   An alterable, associative array containing the structure of the form,
   *   passed by reference.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The form state.
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::buildOptionsForm($form, $form_state);

    foreach ($this->viewsQueryInstances as $instance) {
      $instance->buildOptionsForm($form, $form_state, $this);
    }
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state): void {
    parent::submitOptionsForm($form, $form_state);

    foreach ($this->viewsQueryInstances as $instance) {
      $instance->submitOptionsForm($form, $form_state, $this);
    }
  }

  /**
   * {@inheritdoc}
   *
   * @param bool $get_count
   *   Provide a countQuery if this is true, otherwise provide a normal query.
   */
  public function query($get_count = FALSE): SelectInterface {
    $query = parent::query($get_count);

    foreach ($this->viewsQueryInstances as $instance) {
      $instance->queryAlter($query, $get_count, $this);
    }

    return $query;
  }

  /**
   * {@inheritdoc}
   */
  public function assignEntitiesToResult($ids, array $entities, array $results): array {
    $results = parent::assignEntitiesToResult($ids, $entities, $results);
    foreach ($this->viewsQueryInstances as $instance) {
      $instance->assignEntitiesToResultAlter($ids, $entities, $results);
    }

    return $results;
  }

  /**
   * {@inheritdoc}
   */
  public function calculateDependencies(): array {
    $dependencies = parent::calculateDependencies();

    // Add module dependencies for all views_query plugins used by this query
    // handler. This ensures that exported Views depending on these plugins
    // declare the providing modules as configuration dependencies.
    foreach ($this->viewsQueryInstances as $instance) {
      // Consider the plugin active only if any of its option keys are enabled
      // on this view's query options.
      $is_active = FALSE;
      foreach (array_keys($instance->defineOptions()) as $option_key) {
        if (!empty($this->options[$option_key])) {
          $is_active = TRUE;
          break;
        }
      }

      if ($is_active) {
        $definition = $instance->getPluginDefinition();
        if (!empty($definition['provider'])) {
          $dependencies['module'][] = $definition['provider'];
        }
      }
    }

    if (!empty($dependencies['module'])) {
      $dependencies['module'] = array_values(array_unique($dependencies['module']));
    }

    return $dependencies;
  }

}
