<?php

declare(strict_types=1);

namespace Drupal\views_query;

use Drupal\Component\Plugin\PluginBase;
use Drupal\Core\Database\Query\SelectInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Drupal\Core\StringTranslation\StringTranslationTrait;
use Drupal\views_query\Plugin\views\query\ViewsQuerySql;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Base class for ViewsQuery plugins.
 */
abstract class ViewsQueryPluginBase extends PluginBase implements ViewsQueryInterface, ContainerFactoryPluginInterface {

  use StringTranslationTrait;

  /**
   * Constructs a ViewsQueryPluginBase object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Drupal\Core\Entity\EntityTypeManagerInterface $entityTypeManager
   *   The entity type manager.
   */
  public function __construct(
    array $configuration,
    $plugin_id,
    $plugin_definition,
    protected readonly EntityTypeManagerInterface $entityTypeManager,
  ) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('entity_type.manager'),
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defineOptions(): array {
    return [
      $this->getPluginId() => [
        'default' => FALSE,
      ],
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state, ViewsQuerySql $views_query): void {
    // Auto-generate a checkbox referencing the views_query option.
    $option_key = $this->getPluginId();
    $definition = $this->getPluginDefinition();
    $title = $definition['title'] ?? $this->getPluginId();
    $description = $definition['description'] ?? '';

    $form[$option_key] = [
      '#type' => 'checkbox',
      '#title' => $title,
      '#description' => !empty($description) ? $description : NULL,
      '#default_value' => !empty($views_query->options[$option_key]),
    ];
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state, ViewsQuerySql $views_query): void {}

  /**
   * {@inheritdoc}
   */
  public function queryAlter(SelectInterface $query, bool $get_count, ViewsQuerySql $views_query): void {
    // Centralized enable-check: only delegate when this plugin's option is on.
    if (empty($views_query->options[$this->getPluginId()])) {
      return;
    }
    $this->doQueryAlter($query, $get_count, $views_query);
  }

  /**
   * {@inheritdoc}
   */
  public function assignEntitiesToResultAlter($ids, array $entities, array &$results): void {}

  /**
   * Performs the actual query alteration when the option is enabled.
   *
   * Subclasses should override this method instead of queryAlter() to benefit
   * from the centralized enable-check implemented in the base class.
   *
   * @param \Drupal\Core\Database\Query\SelectInterface $query
   *   The sql query.
   * @param bool $get_count
   *   Provide a countQuery if this is true, otherwise provide a normal query.
   * @param \Drupal\views_query\Plugin\views\query\ViewsQuerySql $views_query
   *   The ViewsQuerySql instance.
   */
  protected function doQueryAlter(SelectInterface $query, bool $get_count, ViewsQuerySql $views_query): void {}

}
