<?php

declare(strict_types=1);

namespace Drupal\tests\views_query\Unit;

use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Messenger\MessengerInterface;
use Drupal\Tests\UnitTestCase;
use Drupal\views\Plugin\views\query\DateSqlInterface;
use Drupal\views_query\Plugin\views\query\ViewsQuerySql;
use Drupal\views_query\ViewsQueryInterface;
use Drupal\views_query\ViewsQueryPluginManager;

/**
 * @covers \Drupal\views_query\Plugin\views\query\ViewsQuerySql::calculateDependencies
 */
final class ViewsQuerySqlDependenciesTest extends UnitTestCase {

  /**
   * Builds a handler with the given plugin instances and options.
   */
  private function buildHandler(array $plugin_instances, array $options = []): ViewsQuerySql {
    $entityTypeManager = $this->createMock(EntityTypeManagerInterface::class);
    $dateSql = $this->createMock(DateSqlInterface::class);
    $messenger = $this->createMock(MessengerInterface::class);

    $manager = $this->createMock(ViewsQueryPluginManager::class);
    $definitions = [];
    foreach ($plugin_instances as $id => $instance) {
      $definitions[$id] = $instance->getPluginDefinition();
    }
    $manager->method('getDefinitions')->willReturn($definitions);
    $manager->method('createInstance')->willReturnCallback(function (string $id) use ($plugin_instances) {
      return $plugin_instances[$id];
    });

    // Create a partial mock to stub out container-dependent parent logic that
    // would otherwise call \Drupal::entityTypeManager() in parent
    // calculateDependencies()->getEntityTableInfo().
    $handler = $this->getMockBuilder(ViewsQuerySql::class)
      ->setConstructorArgs([
        [],
        'views_query',
        [],
        $entityTypeManager,
        $dateSql,
        $messenger,
        $manager,
      ])
      ->onlyMethods(['getEntityTableInfo'])
      ->getMock();

    // Avoid hitting the global container by returning an empty mapping of
    // entity tables; we only want to test the dependencies added by the
    // views_query plugins themselves in this unit test.
    $handler->method('getEntityTableInfo')->willReturn([]);

    // Inject options via reflection, as Views stores them on the handler.
    $ref = new \ReflectionClass($handler);
    $prop = NULL;
    $cur = $ref;
    while ($cur && !$prop) {
      if ($cur->hasProperty('options')) {
        $prop = $cur->getProperty('options');
        break;
      }
      $cur = $cur->getParentClass();
    }
    $this->assertNotNull($prop, 'Could not reflect options property.');
    $prop->setAccessible(TRUE);
    $prop->setValue($handler, $options);

    return $handler;
  }

  /**
   * Creates a mock plugin with the given ID and provider.
   */
  private function mockPlugin(string $id, string $provider, array $option_keys): ViewsQueryInterface {
    $plugin = $this->createMock(ViewsQueryInterface::class);
    $plugin->method('defineOptions')->willReturn(array_fill_keys($option_keys, ['default' => FALSE]));
    $plugin->method('getPluginDefinition')->willReturn([
      'id' => $id,
      'provider' => $provider,
    ]);
    return $plugin;
  }

  /**
   * @covers \Drupal\views_query\Plugin\views\query\ViewsQuerySql::calculateDependencies
   */
  public function testNoActivePluginsAddsNoDeps(): void {
    $foo = $this->mockPlugin('test_foo', 'views_query_test', ['test_foo']);
    $bar = $this->mockPlugin('test_bar', 'views_query_test_alt', ['test_bar']);

    $handler = $this->buildHandler([
      'test_foo' => $foo,
      'test_bar' => $bar,
    ], [
      // All disabled.
      'test_foo' => 0,
      'test_bar' => 0,
    ]);

    $deps = $handler->calculateDependencies();

    // Parent may add some modules (e.g., 'views'), but should not include our
    // providers.
    $this->assertNotContains('views_query_test', $deps['module'] ?? []);
    $this->assertNotContains('views_query_test_alt', $deps['module'] ?? []);
  }

  /**
   * @covers \Drupal\views_query\Plugin\views\query\ViewsQuerySql::calculateDependencies
   */
  public function testSingleActivePluginAddsItsProvider(): void {
    $foo = $this->mockPlugin('test_foo', 'views_query_test', ['test_foo']);
    $bar = $this->mockPlugin('test_bar', 'views_query_test_alt', ['test_bar']);

    $handler = $this->buildHandler([
      'test_foo' => $foo,
      'test_bar' => $bar,
    ], [
      'test_foo' => 1,
      'test_bar' => 0,
    ]);

    $deps = $handler->calculateDependencies();

    $this->assertContains('views_query_test', $deps['module']);
    $this->assertNotContains('views_query_test_alt', $deps['module']);
  }

  /**
   * @covers \Drupal\views_query\Plugin\views\query\ViewsQuerySql::calculateDependencies
   */
  public function testMultipleActivePluginsAddedOnceEach(): void {
    $foo = $this->mockPlugin('test_foo', 'views_query_test', ['test_foo']);
    $bar = $this->mockPlugin('test_bar', 'views_query_test_alt', ['test_bar']);

    $handler = $this->buildHandler([
      'test_foo' => $foo,
      'test_bar' => $bar,
    ], [
      'test_foo' => 1,
      'test_bar' => 1,
    ]);

    $deps = $handler->calculateDependencies();

    $modules = $deps['module'];
    $this->assertContains('views_query_test', $modules);
    $this->assertContains('views_query_test_alt', $modules);
    // Ensure unique providers.
    $this->assertSameSize(array_unique($modules), $modules);
  }

}
