<?php

namespace Drupal\views_scored_sort\Helper;

use Drupal\Component\Datetime\Time;
use Drupal\Core\Routing\CurrentRouteMatch;
use Drupal\node\NodeInterface;
use Drupal\views\ViewExecutable;

/**
 * Reorders views based on custom sort criteria.
 *
 * Pass in $view information and $scoring_fields, information about the fields
 * that were scored in the config, and reorder them based on the scores given
 * to each field.
 */
class RelatedContentScorer {

  /**
   * The current route match.
   *
   * @var \Drupal\Core\Routing\CurrentRouteMatch
   */
  protected $currentRouteMatch;

  /**
   * Time object.
   *
   * @var \Drupal\Component\Datetime\Time
   */
  protected $time;

  /**
   * Constructs a new object.
   *
   * @param \Drupal\Core\Routing\CurrentRouteMatch $currentRouteMatch
   *   The current route match.
   * @param \Drupal\Component\Datetime\Time $time
   *   Time object.
   */
  public function __construct(CurrentRouteMatch $currentRouteMatch, Time $time) {
    $this->currentRouteMatch = $currentRouteMatch;
    $this->time = $time;
  }

  /**
   * Scores and sorts the View results based on topic and published_at.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view executable object.
   * @param array $scoring_fields
   *   An array of fields that have a score in the scored_sort config.
   * @param int|null $originalLimit
   *   Integer representing limit placed on pager.
   */
  public function scoreAndSort(ViewExecutable $view, array $scoring_fields, ?int $originalLimit = NULL) {
    // Get the field definition from the current display.
    $field_definitions = $view->display_handler->getOption('fields');
    // Get the node machine name of the field in case this is an alias.
    $real_fields = [];
    foreach ($scoring_fields as $field_name => $config) {
      if (isset($field_definitions[$field_name])) {
        $real_field_key = $field_definitions[$field_name]['field'];
        $real_fields[$real_field_key] = $config;
      }
    }
    $route_node = $this->currentRouteMatch->getParameter('node');
    $current_nid = $route_node instanceof NodeInterface ? $route_node->id() : NULL;
    if ($route_node instanceof NodeInterface) {
      $taxonomy_reference_fields = [];

      /* If this is an entity reference field, get some information about the
      field. */
      foreach ($real_fields as $field_name => &$config) {
        if (isset($config['type']) && $config['type'] === 'entity_reference_label') {
          $taxonomy_reference_fields[] = [
            'name' => $field_name,
            'score' => $config['score'] ?? 0,
            'match_required' => $config['match_required'] ?? FALSE,
          ];
        }
        /* If this is a datetime field, and the mode is relativity, check the
        current node for the same field so we can save the value. */
        if (isset($config['type'])) {
          if (in_array($config['type'], ['timestamp', 'datetime_default'])) {
            if ($config['mode'] == 'relativity') {
              if ($route_node->hasField($field_name) && !$route_node->get($field_name)->isEmpty()) {
                $config['route_node_timestamp'] = $route_node->get($field_name)->value;
              }
            }
          }
        }
      }
      unset($config);
    }

    foreach ($view->result as $key => $row) {
      /** @var \Drupal\node\NodeInterface $row_node */
      $row_node = $row->_entity;
      $score = 0.0;

      if ($route_node instanceof NodeInterface) {
        // Remove the current node from the view.
        if ($current_nid && $row_node->id() == $current_nid) {
          unset($view->result[$key]);
          continue;
        }

        // Add points for every matching taxonomy term.
        foreach ($taxonomy_reference_fields as $taxonomy_reference_field) {
          if (!empty($taxonomy_reference_field) && $row_node->hasField($taxonomy_reference_field['name'])) {
            $route_node_tags = array_column($route_node->get($taxonomy_reference_field['name'])->getValue() ?? [], 'target_id');
            $row_tags = array_column($row_node->get($taxonomy_reference_field['name'])->getValue() ?? [], 'target_id');
            $matches = 0;
            foreach ($row_tags as $tid) {
              if (in_array($tid, $route_node_tags)) {
                $score += $taxonomy_reference_field['score'];
                $matches += 1;
              }
            }
          }
          /* If this field requires at least one match, and there are none in
          the row, remove the row from the results list. */
          if ($taxonomy_reference_field['match_required'] === 1) {
            if ($matches < 1) {
              unset($view->result[$key]);
            }
          }
        }
      }

      // Penalize older content using computeTimestampScore().
      foreach ($real_fields as $field_name => $config) {
        if (isset($config['type'])) {
          if (in_array($config['type'], ['timestamp', 'datetime_default'])) {
            $score += $this->computeTimestampScore($row_node, $field_name, $config);
          }
        }
      }

      $row->fields['score'] = $score;
      $row->score = $score;
    }

    // Sort rows by computed score (descending)
    usort($view->result, fn($a, $b) => $b->score <=> $a->score);

    // Have to reindex.
    // See https://drupal.stackexchange.com/questions/320639/sorting-a-view-in-hook-views-pre-render-failing-in-final-display
    foreach ($view->result as $key => &$row) {
      $row->index = $key;
    }

    $result = $view->result;
    return $result;
  }

  /**
   * Penalizes content with older timestamps.
   *
   * The score configuration acts as a numerator and the months a denominator
   * to determine how much to penalize older content.
   */
  private function computeTimestampScore(NodeInterface $node, string $field_name, array $config): float {
    if (
      !$node->hasField($field_name) ||
      $node->get($field_name)->isEmpty() ||
      !isset($config['months_decay'], $config['score'])
    ) {
      return 0.0;
    }

    $field = $node->get($field_name);
    if ($field->isEmpty()) {
      return 0.0;
    }
    $value = $field->getValue()[0]['value'];

    if (is_numeric($value)) {
      // Timestamp, use setTimestamp.
      $row_date = (new \DateTime())->setTimestamp((int) $value);
    }
    else {
      // Datetime field, just create object from there.
      $row_date = new \DateTime($value);
    }
    /* If there's a route_node_timestamp, we are sorting by relativity, set
    $now to that timestamp. */
    if (isset($config['route_node_timestamp'])) {
      $raw = $config['route_node_timestamp'];
      if (is_numeric($raw)) {
        $route_date = (new \DateTime())->setTimestamp((int) $raw);
      }
      else {
        $route_date = new \DateTime($raw);
      }
    }
    // Otherwise just use the current date for a recency sort.
    else {
      $route_date = new \DateTime();
    }

    $days_elapsed = $row_date->diff($route_date)->days;
    // The number of days per penalization. E.g punish 10 points every 6 months.
    $decay_days = ((int) $config['months_decay']) * 30.44;

    // Read penalty from the field's configured base score.
    $penalty_per_decay = abs((float) $config['score']);
    $decay_units = $days_elapsed / $decay_days;

    $adjusted_score = ((float) $config['score']) - ($penalty_per_decay * $decay_units);

    return $adjusted_score;
  }

}
