<?php

namespace Drupal\views_scored_sort\Form;

use Drupal\Core\Config\ConfigFactoryInterface;
use Drupal\Core\Config\TypedConfigManagerInterface;
use Drupal\Core\Entity\EntityTypeManagerInterface;
use Drupal\Core\Form\ConfigFormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views\ViewExecutableFactory;
use Symfony\Component\DependencyInjection\ContainerInterface;

/**
 * Implements the related content scoring configuration form.
 */
class RelatedContentScoringConfigForm extends ConfigFormBase {

  /**
   * Entity type manager.
   *
   * @var \Drupal\Core\Entity\EntityTypeManagerInterface
   */
  protected $entityTypeManager;

  /**
   * View executable factory.
   *
   * @var \Drupal\views\ViewExecutableFactory
   */
  protected $viewExecutableFactory;

  /**
   * Constructs a new object.
   */
  public function __construct(
    ConfigFactoryInterface $config_factory,
    TypedConfigManagerInterface $typed_config_manager,
    EntityTypeManagerInterface $entity_type_manager,
    ViewExecutableFactory $viewExecutableFactory,
  ) {
    parent::__construct($config_factory, $typed_config_manager);
    $this->entityTypeManager = $entity_type_manager;
    $this->viewExecutableFactory = $viewExecutableFactory;
  }

  /**
   * Constructs a new object.
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('config.factory'),
      $container->get('config.typed'),
      $container->get('entity_type.manager'),
      $container->get('views.executable')
    );
  }

  /**
   * Set the form id.
   */
  public function getFormId(): string {
    return 'views_scored_sort_scoring_config_form';
  }

  /**
   * Set the configuration id.
   */
  public function getEditableConfigNames(): array {
    return ['views_scored_sort.scoring_config'];
  }

  /**
   * Build the form.
   */
  public function buildForm(array $form, FormStateInterface $form_state): array {
    $view_storage = $this->entityTypeManager->getStorage('view');
    $all_views = $view_storage->loadMultiple();
    $config = $this->config('views_scored_sort.scoring_config');
    $selected_view = $form_state->getValue('view_id') ?? '';
    $selected_display = $form_state->getValue('display_id') ?? '';

    // --- 1. View Dropdown ---
    $view_options = [];
    foreach ($all_views as $id => $view) {
      $view_options[$id] = $view->label();
    }

    $form['view_id'] = [
      '#type' => 'select',
      '#title' => 'View',
      '#options' => $view_options,
      '#empty_option' => '- Select -',
      '#default_value' => $selected_view,
      '#ajax' => [
        'callback' => '::updateDisplays',
        'wrapper' => 'display-wrapper',
      ],
    ];

    // --- 2. Display Dropdown ---
    $form['display_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'display-wrapper'],
    ];

    $display_options = [];
    if ($selected_view) {
      $view_entity = $this->entityTypeManager->getStorage('view')->load($selected_view);
      if ($view_entity) {
        foreach ($view_entity->get('display') as $id => $display) {
          $display_options[$id] = $display['display_title'];
        }
      }
    }

    $form['display_wrapper']['display_id'] = [
      '#type' => 'select',
      '#title' => 'Display',
      '#options' => $display_options,
      '#empty_option' => '- Select -',
      '#default_value' => $selected_display,
      '#ajax' => [
        'callback' => '::updateFields',
        'wrapper' => 'fields-wrapper',
      ],
    ];

    // --- 3. Fields Section ---
    $form['fields_wrapper'] = [
      '#type' => 'container',
      '#attributes' => ['id' => 'fields-wrapper'],
    ];

    if ($selected_view && $selected_display) {
      $view_config = $this->entityTypeManager->getStorage('view')->load($selected_view);
      $view = $this->viewExecutableFactory->get($view_config);
      $view->initDisplay();
      $view->setDisplay($selected_display);
      $fields = $view->getDisplay()->getOption('fields') ?? [];
      $field_definitions = $view->display_handler->getOption('fields');

      $stored_config = $config->get("views.$selected_view.$selected_display") ?? [];

      $form['fields_wrapper'] = [
        '#type' => 'container',
        '#tree' => TRUE,
      ];

      foreach ($fields as $field_id => $info) {
        $field_name = $info['id'] ?? $field_id;
        if (isset($info['type'])) {
          $type = $info['type'];
        }
        else {
          continue;
        }

        // Check the node's field machine name, in case this is an alias.
        $views_field = $field_definitions[$field_id];
        $real_field = $views_field['field'];

        if ($info['type'] == 'entity_reference_label') {
          $field_config = $this->entityTypeManager
            ->getStorage('field_storage_config')
            ->load("node.$real_field");
        }

        if (isset($field_config)) {
          $target_type = $field_config->get('settings')['target_type'];
        }

        $field_key = $field_name;

        if (
          in_array($type, ['datetime', 'timestamp', 'datetime_default']) ||
          ($type === 'entity_reference_label' && $target_type === 'taxonomy_term')
        ) {
          $default = $stored_config[$field_key] ?? [];

          $form['fields_wrapper'][$field_key] = [
            '#type' => 'fieldset',
            '#title' => $this->t('Field: @name', ['@name' => $field_key]),
          ];

          $form['fields_wrapper'][$field_key]['score'] = [
            '#type' => 'number',
            '#title' => 'Score',
            '#default_value' => $default['score'] ?? '',
            '#step' => 0.1,
          ];

          // Store the field type along with the form item.
          $form['fields_wrapper'][$field_key]['type'] = [
            '#type' => 'value',
            '#value' => $type,
          ];

          if ($type === 'datetime' || $type === 'timestamp' || $type === 'datetime_default') {
            $form['fields_wrapper'][$field_key]['score']['#max'] = 0;
            $form['fields_wrapper'][$field_key]['score']['#description'] = 'Subtract
            points for older content by setting the amount of score to be
            subtracted for the specified time period. E.g -3 points every one
            year. Only negative numbers allowed.';
            $form['fields_wrapper'][$field_key]['score']['#title'] = 'Penalty per
            decay.';
            $form['fields_wrapper'][$field_key]['months_decay'] = [
              '#type' => 'number',
              '#title' => 'Months per decay',
              '#default_value' => $default['months_decay'] ?? 6,
              '#step' => 1,
            ];
            $form['fields_wrapper'][$field_key]['mode'] = [
              '#type' => 'select',
              '#title' => t('Scoring mode'),
              '#options' => [
                'recency' => t('Recency'),
                'relativity' => t('Relativity'),
              ],
              '#default_value' => $default['mode'] ?? 'recency',
            ];
          }

          if ($type === 'entity_reference_label' && $target_type === 'taxonomy_term') {
            // Scoring mode dropdown. Doesn't do anything yet.
            $form['fields_wrapper'][$field_key]['mode'] = [
              '#type' => 'select',
              '#title' => t('Scoring mode'),
              '#options' => [
                'match' => t('Match terms'),
              ],
              '#default_value' => $default['mode'] ?? 'match',
            ];
            // Toggle to require at least one match to display the result.
            $form['fields_wrapper'][$field_key]['match_required'] = [
              '#type' => 'checkbox',
              '#title' => t('Match Required?'),
              '#description' => t('Check this box to only show results if there
              is at least one match of this term'),
              '#default_value' => $default['match_required'] ?? FALSE,
            ];
          }
        }
      }
    }

    return parent::buildForm($form, $form_state);
  }

  /**
   * Callback to update display dropdown.
   */
  public function updateDisplays(array &$form, FormStateInterface $form_state): array {
    // Return only the display dropdown container.
    return $form['display_wrapper'];
  }

  /**
   * Callback to update field list based on display selection.
   */
  public function updateFields(array &$form, FormStateInterface $form_state): array {
    // Return only the fieldset area.
    return $form['fields_wrapper'];
  }

  /**
   * Form submission handler.
   */
  public function submitForm(array &$form, FormStateInterface $form_state): void {
    parent::submitForm($form, $form_state);

    $view_id = $form_state->getValue('view_id');
    $display_id = $form_state->getValue('display_id');
    $fields_data = $form_state->getValue(['fields_wrapper']) ?? [];

    $config_data = [];

    foreach ($fields_data as $field_name => $field_config) {
      if (!isset($field_config['score']) || $field_config['score'] === '') {
        // Skip empty.
        continue;
      }

      $entry = ['score' => (float) $field_config['score']];

      if (!empty($field_config['direction'])) {
        $entry['direction'] = $field_config['direction'];
      }
      if (!empty($field_config['mode'])) {
        $entry['mode'] = $field_config['mode'];
      }
      if (!empty($field_config['match_required'])) {
        $entry['match_required'] = $field_config['match_required'];
      }
      if (!empty($field_config['months_decay'])) {
        $entry['months_decay'] = (int) $field_config['months_decay'];
      }

      $entry['type'] = $form_state->getValue(['fields_wrapper', $field_name, 'type']);

      $config_data[$field_name] = $entry;
    }

    $this->configFactory->getEditable('views_scored_sort.scoring_config')
      ->set("views.$view_id.$display_id", $config_data)
      ->save();

    $this->messenger()->addMessage($this->t('Scoring configuration saved for %view / %display.', [
      '%view' => $view_id,
      '%display' => $display_id,
    ]));
  }

}
