<?php

namespace Drupal\views_share\Controller;

use Drupal\Component\Utility\Crypt;
use Drupal\Core\Ajax\AjaxResponse;
use Drupal\Core\Ajax\OpenModalDialogCommand;
use Drupal\Core\Asset\AssetResolverInterface;
use Drupal\Core\Asset\LibraryDependencyResolverInterface;
use Drupal\Core\Asset\LibraryDiscoveryInterface;
use Drupal\Core\Cache\CacheableMetadata;
use Drupal\Core\Controller\ControllerBase;
use Drupal\Core\Language\LanguageInterface;
use Drupal\Core\Language\LanguageManagerInterface;
use Drupal\Core\Render\BareHtmlPageRendererInterface;
use Drupal\Core\Render\BubbleableMetadata;
use Drupal\Core\Render\HtmlResponse;
use Drupal\Core\Render\RenderContext;
use Drupal\Core\StringTranslation\TranslatableMarkup;
use Drupal\Core\Theme\ThemeManagerInterface;
use Drupal\views\ViewExecutable;
use Drupal\views\Views;
use Drupal\views_share\Utility\ViewsShareHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\RequestStack;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\JsonResponse;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Form\FormBuilderInterface;

/**
 * Controller for handling view share, embed, oEmbed, and modal actions.
 */
class ViewsShareController extends ControllerBase {

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The form builder service.
   *
   * @var \Drupal\Core\Form\FormBuilderInterface
   */
  protected $formBuilder;

  /**
   * The asset resolver service.
   *
   * @var \Drupal\Core\Asset\AssetResolverInterface
   */
  protected $assetResolver;

  /**
   * The library discovery service.
   *
   * @var \Drupal\Core\Asset\LibraryDiscoveryInterface
   */
  protected $libraryDiscovery;

  /**
   * The library dependency resolver service.
   *
   * @var \Drupal\Core\Asset\LibraryDependencyResolverInterface
   */
  protected $libraryDependencyResolver;

  /**
   * The theme manager service.
   *
   * @var \Drupal\Core\Theme\ThemeManagerInterface
   */
  protected $themeManager;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * The language manager service.
   *
   * @var \Drupal\Core\Language\LanguageManagerInterface
   */
  protected $languageManager;

  /**
   * The bare HTML page renderer service.
   *
   * @var \Drupal\Core\Render\BareHtmlPageRendererInterface
   */
  protected $bareHtmlPageRenderer;

  /**
   * Constructs a ViewsShareController object.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Form\FormBuilderInterface $formBuilder
   *   The form builder service.
   * @param \Drupal\Core\Asset\AssetResolverInterface $assetResolver
   *   The asset resolver service.
   * @param \Drupal\Core\Asset\LibraryDiscoveryInterface $libraryDiscovery
   *   The library discovery service.
   * @param \Drupal\Core\Asset\LibraryDependencyResolverInterface $libraryDependencyResolver
   *   The library dependency resolver service.
   * @param \Drupal\Core\Theme\ThemeManagerInterface $themeManager
   *   The theme manager service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack service.
   * @param \Drupal\Core\Language\LanguageManagerInterface $languageManager
   *   The language manager service.
   * @param \Drupal\Core\Render\BareHtmlPageRendererInterface $bareHtmlPageRenderer
   *   The bare HTML page renderer service.
   */
  public function __construct(
    RendererInterface $renderer,
    FormBuilderInterface $formBuilder,
    AssetResolverInterface $assetResolver,
    LibraryDiscoveryInterface $libraryDiscovery,
    LibraryDependencyResolverInterface $libraryDependencyResolver,
    ThemeManagerInterface $themeManager,
    RequestStack $requestStack,
    LanguageManagerInterface $languageManager,
    BareHtmlPageRendererInterface $bareHtmlPageRenderer,
  ) {
    $this->renderer = $renderer;
    $this->formBuilder = $formBuilder;
    $this->assetResolver = $assetResolver;
    $this->libraryDiscovery = $libraryDiscovery;
    $this->libraryDependencyResolver = $libraryDependencyResolver;
    $this->themeManager = $themeManager;
    $this->requestStack = $requestStack;
    $this->languageManager = $languageManager;
    $this->bareHtmlPageRenderer = $bareHtmlPageRenderer;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer'),
      $container->get('form_builder'),
      $container->get('asset.resolver'),
      $container->get('library.discovery'),
      $container->get('library.dependency_resolver'),
      $container->get('theme.manager'),
      $container->get('request_stack'),
      $container->get('language_manager'),
      $container->get('bare_html_page_renderer')
    );
  }

  /**
   * Handles the preview embedding of a view.
   *
   * @param string $view_id
   *   The machine name of the view.
   * @param string $display_id
   *   The display ID.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The rendered view preview.
   */
  public function preview(string $view_id, string $display_id, Request $request): Response {
    $view = Views::getView($view_id);
    if (!$view) {
      return new Response('View not found.', 404);
    }

    $view->setDisplay($display_id);
    // Retrieve additional arguments if necessary.
    $query = $request->query->all();
    $view->setArguments($query['views_share_args'] ?? []);
    $share_plugin_options = $this->shareHandlerOptions($view, $display_id);
    $this->processHandlerOptions($view, $display_id, $share_plugin_options);
    $view->preExecute();
    $view->execute();

    $render_array = [
      '#theme' => 'views_share_preview',
      '#view' => $view,
      '#title' => new TranslatableMarkup('Customize and preview embedded @title', [
        '@title' => $view->getTitle(),
      ]),
      '#form' => $this->formBuilder->getForm('Drupal\\views_share\\Form\\ViewsSharePreviewForm', $view),
    ];

    return $this->buildViewResponse($view, $render_array, 'preview');
  }

  /**
   * Handles embedding of a view.
   *
   * @param string $view_id
   *   The machine name of the view.
   * @param string $display_id
   *   The display ID.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The rendered embedded view.
   */
  public function embed(string $view_id, string $display_id, Request $request): Response {
    $view = Views::getView($view_id);
    if (!$view || !$view->access($display_id)) {
      return new Response('Not found', 404);
    }

    $view->setDisplay($display_id);
    // Retrieve additional arguments if necessary.
    $query = $request->query->all();
    $view->setArguments($query['views_share_args'] ?? []);
    $share_plugin_options = $this->shareHandlerOptions($view, $display_id);
    $this->processHandlerOptions($view, $display_id, $share_plugin_options);
    $view->preExecute();
    $view->execute();
    $content = $view->render();

    $render_array = [
      '#theme' => 'views_share_embed',
      '#view' => $view,
      '#title' => $view->getTitle(),
      '#content' => $content,
    ];
    $render_array['#attached']['drupalSettings']['viewsShareEmbed']['rewriteLinks'] = $share_plugin_options['rewrite_links'];

    return $this->buildViewResponse($view, $render_array, 'embed');
  }

  /**
   * Helper method to build and return a response for a view.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view executable.
   * @param array $render_array
   *   The render array to use as a base.
   * @param string $type
   *   The type of response (preview or embed).
   *
   * @return \Drupal\Core\Render\HtmlResponse
   *   The HTML response.
   */
  protected function buildViewResponse(ViewExecutable $view, array $render_array, string $type): HtmlResponse {
    $response = new HtmlResponse('', Response::HTTP_OK, [
      'Content-Type' => 'text/html; charset=UTF-8',
    ]);

    // Get current language information.
    $language = $this->languageManager->getCurrentLanguage();

    // Generate a random placeholder token.
    $placeholder_token = Crypt::randomBytesBase64(55);

    // Add common elements to the render array.
    $render_array['#base_url'] = $this->requestStack->getCurrentRequest()->getSchemeAndHttpHost();
    $render_array['#language'] = $language->getId();
    $render_array['#language_dir'] = $language->getDirection() === LanguageInterface::DIRECTION_RTL ? 'rtl' : 'ltr';
    $render_array['#placeholder_token'] = $placeholder_token;

    // Add libraries.
    $render_array['#attached']['library'] = $this->getLibraries($type);

    // Check if BigPipe is enabled.
    $bigpipe_enabled = $this->moduleHandler()->moduleExists('big_pipe');
    if ($bigpipe_enabled) {
      // Add required cache contexts.
      if (!isset($render_array['#cache']['contexts'])) {
        $render_array['#cache']['contexts'] = [];
      }
      $render_array['#cache']['contexts'][] = 'cookies:big_pipe_nojs';
    }

    // Create placeholder strings for these keys.
    $types = [
      'styles' => 'css',
      'scripts' => 'js',
      'scripts_bottom' => 'js-bottom',
      'head' => 'head',
    ];
    foreach ($types as $type_key => $placeholder_name) {
      $placeholder = '<' . $placeholder_name . '-placeholder token="' . $placeholder_token . '">';
      $render_array['#attached']['html_response_attachment_placeholders'][$type_key] = $placeholder;
    }

    // Process attachments by the bare HTML page renderer if Core >= 11.
    if (version_compare(\Drupal::VERSION, '11.0.0', '>=')) {
      $this->bareHtmlPageRenderer->systemPageAttachments($render_array);
    }

    // Render in a context to capture bubbleable .
    $context = new RenderContext();
    $content = $this->renderer->executeInRenderContext($context, function () use ($render_array) {
      return $this->renderer->render($render_array);
    });

    $response
      ->setContent($content)
      ->addCacheableDependency($view)
      ->addCacheableDependency(CacheableMetadata::createFromRenderArray($render_array));

    // Process bubbleable metadata from the render context.
    $this->processBubbleableMetadata($context, $response);

    return $response;
  }

  /**
   * Process bubbleable metadata from the render context and add to response.
   *
   * @param \Drupal\Core\Render\RenderContext $context
   *   The render context.
   * @param \Drupal\Core\Render\HtmlResponse $response
   *   The response to add metadata to.
   */
  protected function processBubbleableMetadata(RenderContext $context, HtmlResponse $response): void {
    // Modules and themes implementing hook_views_share_*_preprocess()
    // can add additional #cache and #attachments to a render array. If this
    // occurs, the render context won't be empty, and we need to ensure the
    // added metadata is bubbled up to the response.
    // @see \Drupal\Core\Theme\ThemeManager::render()
    if (!$context->isEmpty()) {
      $bubbleable_metadata = $context->pop();
      assert($bubbleable_metadata instanceof BubbleableMetadata);
      $response->addCacheableDependency($bubbleable_metadata);
      $response->addAttachments($bubbleable_metadata->getAttachments());
    }
  }

  /**
   * Get libraries to include based on the response type.
   *
   * @param string $type
   *   The type of response (preview or embed).
   *
   * @return array
   *   An array of libraries with dependencies resolved.
   */
  protected function getLibraries(string $type): array {
    // Create a list of libraries to include.
    $libraries_to_include = [];

    // Add the Views Share library specific to the type.
    $libraries_to_include[] = 'views_share/' . $type;

    // Resolve all dependencies.
    return $this->libraryDependencyResolver->getLibrariesWithDependencies($libraries_to_include);
  }

  /**
   * Handles oEmbed response for a view.
   *
   * @param string $view_id
   *   The machine name of the view.
   * @param string $display_id
   *   The display ID.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The oEmbed data as JSON.
   */
  public function oembed(string $view_id, string $display_id, Request $request): Response {
    $view = Views::getView($view_id);
    if (!$view || !$view->access($display_id)) {
      return new JsonResponse(['error' => 'Access denied'], 403);
    }

    $query = $request->query->all();

    $view->setDisplay($display_id);
    // Retrieve additional arguments if necessary.
    $view->setArguments($query['views_share_args'] ?? []);
    $handler_options = $this->shareHandlerOptions($view, $display_id);
    $this->processHandlerOptions($view, $display_id, $handler_options);
    $view->preExecute();
    $view->execute();

    $base_url = $request->getSchemeAndHttpHost();

    $width = (int) $handler_options['embed_width'] ?? 600;
    $height = (int) $handler_options['embed_height'] ?? 400;

    $oembed_data = [
      'version' => '1.0',
      'type' => 'rich',
      'width' => $width,
      'height' => $height,
      'title' => $view->getTitle(),
      'url' => ViewsShareHelper::getEmbedUrl($view, $query, TRUE, TRUE),
      'provider_name' => $this->config('system.site')->get('name'),
      'provider_url' => $base_url,
      'html' => ViewsShareHelper::getEmbedCode(
        $view,
        $query,
        $width,
        $height,
        $handler_options['embed_style'] ?? ''
      ),
    ];
    // Add any additional oEmbed data if needed.
    if (!empty($handler_options['oembed_author_name'])) {
      $oembed_data['author_name'] = $handler_options['oembed_author_name'];
    }
    if (!empty($handler_options['oembed_author_url'])) {
      $oembed_data['author_url'] = $handler_options['oembed_author_url'];
    }

    // Get the format to return based on user's selected input.
    $format = $request->get('format');
    if ($format === 'xml') {
      // Convert the oEmbed data to XML format.
      $xml = new \SimpleXMLElement('<oembed/>');
      foreach ($oembed_data as $key => $value) {
        $xml->addChild($key, htmlspecialchars($value));
      }
      return new Response($xml->asXML(), 200, ['Content-Type' => 'application/xml']);
    }

    return new JsonResponse($oembed_data);
  }

  /**
   * Handles rendering a form in a modal window.
   *
   * @param string $view_id
   *   The machine name of the view.
   * @param string $display_id
   *   The display ID.
   * @param \Symfony\Component\HttpFoundation\Request $request
   *   The request.
   *
   * @return \Symfony\Component\HttpFoundation\Response
   *   The rendered modal form.
   *
   * @throws \Exception
   *   If there are issues with rendering.
   */
  public function modal(string $view_id, string $display_id, Request $request): Response {
    $view = Views::getView($view_id);
    if (!$view) {
      return new Response('View not found.', 404);
    }

    $query = $request->query->all();
    $view->setDisplay($display_id);
    // Retrieve additional arguments if necessary.
    $view->setArguments($query['views_share_args'] ?? []);

    // Build the form.
    $form = $this->formBuilder->getForm('Drupal\\views_share\\Form\\ViewsShareForm', $view);
    $rendered_form = $this->renderer->render($form);

    // Build an AjaxResponse that opens a modal dialog.
    $response = new AjaxResponse();
    $response->addCommand(new OpenModalDialogCommand(
      $this->t('Share this view'),
      $rendered_form,
      ['width' => 600]
    ));
    return $response;
  }

  /**
   * Handles the share plugin options.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view executable object.
   * @param string $display_id
   *   The display ID.
   * @param array $share_plugin_options
   *   The options for the share plugin.
   */
  public function processHandlerOptions(ViewExecutable $view, string $display_id, array $share_plugin_options): void {
    // Return if options are empty.
    if (empty($share_plugin_options)) {
      return;
    }

    // Handle header/footer visibility.
    if (!empty($share_plugin_options['hide_header_footer'])) {
      foreach (['header', 'footer'] as $region) {
        $handlers = $view->getHandlers($region, $display_id);
        foreach ($handlers as $handler_id => $handler) {
          if ($handler_id !== 'share') {
            // Remove all handlers except the share handler.
            $view->removeHandler($display_id, $region, $handler_id);
          }
        }
      }
    }

    // Hide attachments if needed.
    if (!empty($share_plugin_options['hide_attachments'])) {
      foreach ($view->displayHandlers as $display_id => $display) {
        if ($display->getPluginId() === 'attachment') {
          // Disable the attachment display.
          $display->setOption('enabled', FALSE);
        }
      }
    }

  }

  /**
   * Retrieves the share plugin options from the view.
   *
   * This method checks both the header and footer regions of the view for
   * the share plugin options.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view executable object.
   * @param string $display_id
   *   The display ID.
   *
   * @return array
   *   The share plugin options, or an empty array if not found.
   */
  public function shareHandlerOptions(ViewExecutable $view, string $display_id): array {
    // Retrieve the share plugin options from the view.
    $share_plugin_options = [];
    foreach (['header', 'footer'] as $region) {
      $handlers = $view->getHandlers($region, $display_id);
      if (!empty($handlers) && array_key_exists('share', $handlers)) {
        $share_plugin_options = $handlers['share'];
        break;
      }
    }

    return $share_plugin_options;
  }

}
