<?php

namespace Drupal\views_share\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Url;
use Drupal\Core\Link;
use Drupal\Core\Render\RendererInterface;
use Drupal\Core\Extension\ModuleHandlerInterface;
use Drupal\views\ViewExecutable;
use Drupal\views_share\Utility\ViewsShareHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a view sharing form rendered in a modal window.
 */
class ViewsShareForm extends FormBase {

  /**
   * The view to share.
   *
   * @var \Drupal\views\ViewExecutable
   */
  protected $view;

  /**
   * The renderer service.
   *
   * @var \Drupal\Core\Render\RendererInterface
   */
  protected $renderer;

  /**
   * The module handler.
   *
   * @var \Drupal\Core\Extension\ModuleHandlerInterface
   */
  protected $moduleHandler;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a new ViewsShareForm.
   *
   * @param \Drupal\Core\Render\RendererInterface $renderer
   *   The renderer service.
   * @param \Drupal\Core\Extension\ModuleHandlerInterface $module_handler
   *   The module handler service.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack service.
   */
  public function __construct(RendererInterface $renderer, ModuleHandlerInterface $module_handler, RequestStack $requestStack) {
    $this->renderer = $renderer;
    $this->moduleHandler = $module_handler;
    $this->requestStack = $requestStack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('renderer'),
      $container->get('module_handler'),
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'views_share_form';
  }

  /**
   * Builds the view sharing form.
   *
   * The $view parameter should be passed in from the caller (typically via the
   * route or as a form parameter). It is expected to be a Drupal views
   * executable.
   *
   * @param array $form
   *   The render array for the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param \Drupal\views\ViewExecutable|null $view
   *   (Optional) The view object to be shared.
   *
   * @return array
   *   The render array for the form.
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?ViewExecutable $view = NULL) {
    if ($view) {
      $this->view = $view;
    }
    if (!$this->view) {
      $form['error'] = [
        '#markup' => $this->t('View not available.'),
      ];
      return $form;
    }

    // Build the view.
    $this->view->build();

    // Get current query parameters.
    $query = $this->requestStack->getCurrentRequest()->query->all();

    // Generate the view URL using the legacy helper.
    // _views_share_view_url() should return an array with the URL and options.
    $url = ViewsShareHelper::getViewUrl($this->view, $query);
    $url_string = $url->toString();

    // Attach our form library (JS and CSS).
    $form['#attached']['library'][] = 'views_share/form';

    // If the 'shorten' module is enabled, add a checkbox and attach settings.
    if ($this->moduleHandler->moduleExists('shorten')) {
      $form['short_url'] = [
        '#type' => 'checkbox',
        '#title' => $this->t('Short URL'),
        '#default_value' => FALSE,
      ];

      // Use shorten_url() if available; otherwise, fall back.
      $short_url = function_exists('shorten_url') ? shorten_url($url_string) : $url_string;

      $form['#attached']['drupalSettings']['viewsShare'] = [
        'originalURL' => $url_string,
        'shortURL' => $short_url,
      ];
    }

    // Share link field.
    $form['share_link'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Link'),
      '#title_display' => 'invisible',
      '#default_value' => ViewsShareHelper::buildShareableUrl($url)->toString(),
      '#size' => 100,
      '#description' => $this->t('Paste link in email or IM.'),
      '#attributes' => [
        'readonly' => 'readonly',
        'class' => ['views-share-link'],
        'title' => $this->view->getTitle(),
      ],
    ];

    // Generate the embed code.
    $plugin = ViewsShareHelper::findPlugin($this->view);
    $iframe = ViewsShareHelper::getEmbedCode(
      $this->view,
      $query,
      (int) $plugin->options['embed_width'],
      (int) $plugin->options['embed_height'],
      $plugin->options['embed_style']
    );
    $form['share_embed'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Embed code'),
      '#title_display' => 'invisible',
      '#default_value' => $iframe,
      '#size' => 100,
      '#description' => $this->t('Paste HTML to embed in website.'),
      '#attributes' => [
        'readonly' => 'readonly',
        'class' => ['views-share-embed'],
      ],
    ];

    // Build share preview link.
    // Remove the wrapper_format query parameter if it exists.
    $query = ViewsShareHelper::removeWrapperFormat($query);
    $preview_url = Url::fromUserInput("/view/{$this->view->id()}/{$this->view->current_display}/preview", ['query' => $query]);
    $link = Link::fromTextAndUrl($this->t('Customize and preview embedded view'), $preview_url)->toString();

    $form['share_preview'] = [
      '#type' => 'markup',
      '#markup' => $link,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // No submission handling is required.
  }

}
