<?php

namespace Drupal\views_share\Form;

use Drupal\Core\Form\FormBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\Core\Render\Markup;
use Drupal\views\ViewExecutable;
use Drupal\views_share\Utility\ViewsShareHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Provides a preview form for customizing the embed code.
 */
class ViewsSharePreviewForm extends FormBase {

  /**
   * The view to be shared.
   *
   * @var \Drupal\views\ViewExecutable
   */
  protected $view;

  /**
   * The request stack service.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected $requestStack;

  /**
   * Constructs a new ViewsSharePreviewForm.
   */
  public function __construct(RequestStack $requestStack) {
    $this->requestStack = $requestStack;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container) {
    return new static(
      $container->get('request_stack')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function getFormId() {
    return 'views_share_preview_form';
  }

  /**
   * Builds the preview form.
   *
   * The $view parameter must be passed in when calling the form.
   *
   * @param array $form
   *   The render array for the form.
   * @param \Drupal\Core\Form\FormStateInterface $form_state
   *   The current form state.
   * @param \Drupal\views\ViewExecutable|null $view
   *   The view object for which the embed preview is being built.
   *
   * @return array
   *   The render array for the form.
   */
  public function buildForm(array $form, FormStateInterface $form_state, ?ViewExecutable $view = NULL) {
    if ($view) {
      $this->view = $view;
    }
    else {
      $form['error'] = [
        '#markup' => $this->t('View not available.'),
      ];
      return $form;
    }

    // Retrieve the plugin that contains the embed options.
    $plugin = ViewsShareHelper::findPlugin($this->view);

    // Embed width and height fields.
    $form['embed_width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Embeddable frame width'),
      '#default_value' => $plugin->options['embed_width'],
      '#field_suffix' => 'px',
      '#size' => 10,
    ];
    $form['embed_height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Embeddable frame height'),
      '#default_value' => $plugin->options['embed_height'],
      '#field_suffix' => 'px',
      '#size' => 10,
    ];

    // Get query parameters from current request.
    $query = $this->requestStack->getCurrentRequest()->query->all();

    // Generate the embed code using current width/height values.
    $iframe = ViewsShareHelper::getEmbedCode(
      $this->view,
      $query,
      (int) $plugin->options['embed_width'],
      (int) $plugin->options['embed_height'],
      $plugin->options['embed_style']
    );

    // Textarea for the embed code.
    $form['embed_share'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Embed code'),
      '#default_value' => $iframe,
      '#rows' => 3,
      '#description' => $this->t('Paste HTML to embed in website.'),
      '#attributes' => [
        'readonly' => 'readonly',
        'class' => ['views-share-embed'],
      ],
    ];

    // Preview area for the embed code.
    $form['embed_preview'] = [
      '#type' => 'item',
      '#title' => $this->t('Preview'),
      '#markup' => Markup::create('<div class="views-share-preview-wrapper">' . $iframe . '</div>'),
    ];

    // Attach the drupalSettings for JS.
    $form['#attached']['drupalSettings']['viewsSharePreview'] = [
      'embedCode' => $iframe,
    ];

    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function submitForm(array &$form, FormStateInterface $form_state) {
    // This form is for preview only; no submit handler is needed.
  }

}
