<?php

declare(strict_types=1);

namespace Drupal\views_share\Plugin\views\area;

use Drupal\Core\Routing\RequestContext;
use Drupal\Core\Link;
use Drupal\Core\Url;
use Drupal\views\Plugin\views\area\AreaPluginBase;
use Drupal\Core\Form\FormStateInterface;
use Drupal\views_share\Utility\ViewsShareHelper;
use Symfony\Component\DependencyInjection\ContainerInterface;
use Drupal\Core\Plugin\ContainerFactoryPluginInterface;
use Symfony\Component\HttpFoundation\RequestStack;

/**
 * Plugin implementation of the "share" area.
 *
 * @ViewsArea("views_share_plugin_area_share")
 */
class ShareArea extends AreaPluginBase implements ContainerFactoryPluginInterface {

  /**
   * The request stack.
   *
   * @var \Symfony\Component\HttpFoundation\RequestStack
   */
  protected RequestStack $requestStack;

  /**
   * The request context.
   *
   * @var \Drupal\Core\Routing\RequestContext
   */
  protected RequestContext $requestContext;

  /**
   * Constructs a new ShareArea object.
   *
   * @param array $configuration
   *   A configuration array containing information about the plugin instance.
   * @param string $plugin_id
   *   The plugin ID for the plugin instance.
   * @param mixed $plugin_definition
   *   The plugin implementation definition.
   * @param \Symfony\Component\HttpFoundation\RequestStack $requestStack
   *   The request stack.
   * @param \Drupal\Core\Routing\RequestContext $request_context
   *   The request context.
   */
  public function __construct(array $configuration, $plugin_id, $plugin_definition, RequestStack $requestStack, RequestContext $request_context) {
    parent::__construct($configuration, $plugin_id, $plugin_definition);
    $this->requestStack = $requestStack;
    $this->requestContext = $request_context;
  }

  /**
   * {@inheritdoc}
   */
  public static function create(ContainerInterface $container, array $configuration, $plugin_id, $plugin_definition) {
    return new static(
      $configuration,
      $plugin_id,
      $plugin_definition,
      $container->get('request_stack'),
      $container->get('router.request_context')
    );
  }

  /**
   * {@inheritdoc}
   */
  public function defineOptions() {
    $options = parent::defineOptions();
    $options['share_label'] = [
      'default' => $this->t('Share this view'),
    ];
    $options['embed_label'] = [
      'default' => $this->t('View original'),
    ];
    $options['embed_link_nofollow'] = [
      'default' => FALSE,
      'type' => 'boolean',
    ];
    $options['embed_width'] = [
      'default' => 400,
      'type' => 'number',
    ];
    $options['embed_height'] = [
      'default' => 250,
      'type' => 'number',
    ];
    $options['embed_style'] = [
      'default' => 'margin:0 auto; border:0;',
    ];
    $options['hide_exposed_filters'] = [
      'default' => FALSE,
      'type' => 'boolean',
    ];
    $options['hide_header_footer'] = [
      'default' => FALSE,
      'type' => 'boolean',
    ];
    $options['hide_attachments'] = [
      'default' => FALSE,
      'type' => 'boolean',
    ];
    $options['rewrite_links'] = [
      'default' => FALSE,
      'type' => 'boolean',
    ];
    $options['oembed'] = [
      'default' => FALSE,
      'type' => 'boolean',
    ];
    $options['oembed_author_name'] = [
      'default' => '',
    ];
    $options['oembed_author_url'] = [
      'default' => '',
    ];

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);
    $form['share_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label for share link'),
      '#description' => $this->t('The label shown on the original view for the button that opens up the sharing dialog.'),
      '#default_value' => $this->options['share_label'],
    ];
    $form['embed_label'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Label for original link'),
      '#description' => $this->t('The label shown on the embedded view for the button that links back to the original view.'),
      '#default_value' => $this->options['embed_label'],
    ];
    $form['embed_width'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Embeddable frame width'),
      '#default_value' => $this->options['embed_width'],
      '#field_suffix' => 'px',
      '#size' => 10,
    ];
    $form['embed_height'] = [
      '#type' => 'textfield',
      '#title' => $this->t('Embeddable frame height'),
      '#default_value' => $this->options['embed_height'],
      '#field_suffix' => 'px',
      '#size' => 10,
    ];
    $form['embed_style'] = [
      '#type' => 'textarea',
      '#title' => $this->t('Embeddable frame style'),
      '#default_value' => $this->options['embed_style'],
    ];
    $form['embed_link_nofollow'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Prevent crawlers from following original links'),
      '#description' => $this->t('Add the <code>rel="nofollow"</code> attribute to links back to the original view to prevent crawlers from following them.'),
      '#default_value' => $this->options['embed_link_nofollow'],
    ];
    $form['hide_exposed_filters'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide exposed filters when embedding'),
      '#default_value' => $this->options['hide_exposed_filters'],
    ];
    $form['hide_header_footer'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide header/footer elements when embedding'),
      '#default_value' => $this->options['hide_header_footer'],
    ];
    $form['hide_attachments'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Hide attachments when embedding'),
      '#default_value' => $this->options['hide_attachments'],
    ];
    $form['rewrite_links'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Rewrite links to open in new tab when embedding'),
      '#default_value' => $this->options['rewrite_links'],
    ];

    // Build the endpoint URL for oEmbed support.
    $query = $this->view->getExposedInput();
    $query['format'] = 'output_format';
    $endpoint = ViewsShareHelper::getEmbedUrl($this->view, $query, TRUE, TRUE);

    $form['oembed'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Provide oEmbed support'),
      '#default_value' => $this->options['oembed'],
      '#description' => $this->t('Generate oEmbed discovery tags on the page and respond to oEmbed calls. The oEmbed endpoint for this display is <a href=":endpoint">:endpoint</a> where <code>output_format</code> can be "json" or "xml".', [':endpoint' => $endpoint]),
    ];
    $form['oembed_author_name'] = [
      '#type' => 'textfield',
      '#title' => $this->t('oEmbed author name'),
      '#default_value' => $this->options['oembed_author_name'],
      '#states' => [
        'visible' => [
          ':input[name="options[oembed]"]' => ['checked' => TRUE],
        ],
      ],
      '#description' => $this->t('Generate an "author_name" entry representing the name of the author/owner of this oEmbed resource (optional).'),
    ];
    $form['oembed_author_url'] = [
      '#type' => 'textfield',
      '#title' => $this->t('oEmbed author URL'),
      '#default_value' => $this->options['oembed_author_url'],
      '#states' => [
        'visible' => [
          ':input[name="options[oembed]"]' => ['checked' => TRUE],
        ],
      ],
      '#description' => $this->t('Generate an "author_url" entry representing the URL of the author/owner of this oEmbed resource (optional).'),
    ];
    return $form;
  }

  /**
   * {@inheritdoc}
   */
  public function render($empty = FALSE) {
    $query = $this->requestStack->getCurrentRequest()->query->all();

    // If the view is in embedded mode, show the original view link.
    if (!empty($this->view->is_embedding)) {
      // Do not render the share button if the hide_header_footer option is set.
      if (!empty($this->options['hide_header_footer'])) {
        return [];
      }

      return $this->renderEmbeddedLink($query);
    }

    // Add oEmbed metadata if enabled.
    if (!empty($this->options['oembed'])) {
      $this->addOembedHead('json', $query);
      $this->addOembedHead('xml', $query);
    }

    return $this->renderShareButton($query);
  }

  /**
   * Renders the embedded link using a Twig template.
   *
   * @param array $query
   *   The current query parameters.
   *
   * @return array
   *   A render array.
   */
  protected function renderEmbeddedLink(array $query): array {
    $url = ViewsShareHelper::getViewUrl($this->view, $query);

    $attributes = $url->getOption('attributes') ?: [];
    $attributes += [
      'target' => '_blank',
      'class' => ['views-share-embedded'],
    ];
    if (!empty($this->options['embed_link_nofollow'])) {
      $attributes['rel'] = 'nofollow';
    }
    $url->setOption('attributes', $attributes);
    $label = $this->options['embed_label'] ?? $this->t('View original');
    // @todo find a better way to detect if we are about to show the original
    // link.
    $url = $label === 'View original' ? ViewsShareHelper::buildShareableUrl($url, TRUE) : $url;
    return [
      '#theme' => 'views_share_embedded_link',
      '#label' => $label,
      '#url' => $url,
      '#link' => Link::fromTextAndUrl($label, $url),
      '#view' => $this->view,
      '#options' => $this->options,
      '#attributes' => $attributes,
    ];
  }

  /**
   * Renders the share button using a Twig template.
   *
   * @param array $query
   *   The current query parameters.
   *
   * @return array
   *   A render array.
   */
  protected function renderShareButton(array $query): array {
    // Build share button link.
    $route_parameters = [
      'view_id' => $this->view->id(),
      'display_id' => $this->view->current_display,
    ];
    $query['referer'] = $this->getReferrerUri();
    $query['views_share_args'] = $this->view->args;

    $title_attr = $this->view->getDisplay()->getOption('title') ?? $this->t('Default Title');
    $url = Url::fromRoute('views_share.share', $route_parameters, [
      'query' => $query,
      'attributes' => [
        'title' => $this->t('Share @name', ['@name' => $title_attr]),
        'class' => [
          'use-ajax',
          'views-share',
        ],
        'data-dialog-type' => 'modal',
        'data-dialog-options' => json_encode([
          'width' => 600,
          'height' => 400,
          'closeText' => $this->t('Close'),
        ]),
        'rel' => 'nofollow',
        'name' => ViewsShareHelper::getHashtag($this->view),
      ],
    ]);

    // Create the build array with the proper structure for AJAX.
    $button_text = $this->options['share_label'] ?? $this->t('Share this view');
    $build = [
      '#theme' => 'views_share_button',
      '#label' => $button_text,
      '#url' => $url,
      '#link' => Link::fromTextAndUrl($button_text, $url),
      '#view' => $this->view,
      '#options' => $this->options,
      '#attributes' => [],
      '#hashtag' => ViewsShareHelper::getHashtag($this->view),
    ];

    // Attach libraries and settings for the modal.
    $build['#attached']['library'][] = 'views_share/modal';
    $build['#attached']['drupalSettings']['views_share'] = [
      'modalSize' => [
        'type' => 'scaled',
        'width' => 500,
        'height' => 240,
        'addWidth' => 0,
        'addHeight' => 0,
        'contentRight' => 0,
        'contentBottom' => 0,
      ],
      'modalTheme' => 'viewsShareModal',
      'modalOptions' => [
        'opacity' => 0.5,
        'background-color' => '#FFF',
      ],
      'animation' => 'fadeIn',
      'animationSpeed' => 'normal',
      'closeText' => $this->t('Close'),
    ];

    return $build;
  }

  /**
   * Adds an oEmbed link tag to the HTML head.
   *
   * @param string $format
   *   The format of the oEmbed response (e.g., 'json' or 'xml').
   * @param array $query
   *   The query parameters.
   */
  protected function addOembedHead(string $format, array $query) {
    $oembed_query = $query;
    $oembed_query['format'] = $format;
    $endpoint = ViewsShareHelper::getEmbedUrl($this->view, $oembed_query, TRUE, TRUE);
    $element = [
      '#tag' => 'link',
      '#attributes' => [
        'rel' => 'alternate',
        'type' => "application/{$format}+oembed",
        'title' => $this->view->getTitle(),
        'href' => $endpoint,
      ],
    ];
    // Attach the element to the page head.
    $this->view->element['#attached']['html_head'][] = [$element, "oembed_{$format}_" . $this->view->id()];
  }

  /**
   * Gets the referrer URI.
   *
   * @return string
   *   The referrer URI.
   */
  protected function getReferrerUri(): string {
    $current_request = $this->requestStack->getCurrentRequest();
    if ($this->isAjax()) {
      $uri = $current_request->headers->get('referer');
      $base_url = $this->requestContext->getCompleteBaseUrl();
      if ($base_url !== '' && str_starts_with($uri, $base_url)) {
        $uri = substr($uri, strlen($base_url));
      }
      $parts = parse_url($uri);
      return empty($parts['path']) ? '/' : $parts['path'];
    }
    return $current_request->getRequestUri();
  }

  /**
   * Determines if the current request is via AJAX.
   *
   * @return bool
   *   TRUE if the current request is via AJAX, FALSE otherwise.
   */
  protected function isAjax(): bool {
    return $this->requestStack->getCurrentRequest()->get('_route') === 'views.ajax';
  }

}
