<?php

declare(strict_types=1);

namespace Drupal\views_share\Utility;

use Drupal\Core\Url;
use Drupal\Component\Utility\Html;
use Drupal\views\ViewExecutable;
use Drupal\views_share\Plugin\views\area\ShareArea;

/**
 * Provides helper methods for generating embed code, URLs, and related values.
 */
class ViewsShareHelper {

  /**
   * Returns the embed code (an iframe) for a given view.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view to embed.
   * @param array $query
   *   Query parameters to attach.
   * @param int $width
   *   The width in pixels for the iframe.
   * @param int $height
   *   The height in pixels for the iframe.
   * @param string $style
   *   Additional inline CSS styles.
   *
   * @return string
   *   The complete iframe HTML string.
   */
  public static function getEmbedCode(ViewExecutable $view, array $query, int $width, int $height, string $style): string {
    // Remove the wrapper_format query parameter if it exists.
    $query = self::removeWrapperFormat($query);
    $embed_url = self::getEmbedUrl($view, $query, TRUE);

    // Build the iframe HTML.
    return '<iframe src="' . $embed_url . '" class="views-share-iframe" style="width:' . $width . 'px; height:' . $height . 'px; ' . $style . '"></iframe>';
  }

  /**
   * Returns the embed URL for a view.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view to embed.
   * @param array $query
   *   Query parameters to attach.
   * @param bool $absolute
   *   Whether to generate an absolute URL.
   * @param bool $oembed
   *   Whether this URL is for oEmbed discovery.
   *
   * @return string
   *   The embed URL as a string.
   */
  public static function getEmbedUrl(ViewExecutable $view, array $query = [], bool $absolute = FALSE, bool $oembed = FALSE): string {
    $embed = $oembed ? 'oembed' : 'embed';
    // Build the path. Prepend a slash to ensure correct parsing.
    $path = "/view/{$view->id()}/{$view->current_display}/{$embed}";
    // Remove the wrapper_format query parameter if it exists.
    $query = self::removeWrapperFormat($query);
    $options = [
      'query' => $query,
      'absolute' => $absolute,
    ];

    return Url::fromUserInput($path, $options)->toString();
  }

  /**
   * Returns a unique hashtag (CSS-friendly) for a view.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view for which to generate a hashtag.
   *
   * @return string
   *   The cleaned hashtag string.
   */
  public static function getHashtag(ViewExecutable $view): string {
    return Html::cleanCssIdentifier($view->id() . '--' . $view->current_display);
  }

  /**
   * Returns the view URL and additional options for use with Url or Link.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view object.
   * @param array $query
   *   Query parameters to attach.
   *
   * @return \Drupal\Core\Url
   *   The URL.
   */
  public static function getViewUrl(ViewExecutable $view, array $query): Url {
    // Check if a referer is available in the current request.
    $request = \Drupal::request();
    $referer = $request->query->get('referer');
    $url = $referer ?
      Url::fromUserInput($referer) :
      $view->getUrl();

    // Remove the wrapper_format query parameter if it exists.
    $query = self::removeWrapperFormat($query);
    $options = ['query' => $query, 'absolute' => TRUE];
    // If the display is an attachment, add a fragment with the view hashtag.
    if ($view->getDisplay()->getPluginId() === 'attachment') {
      $options['fragment'] = self::getHashtag($view);
    }

    return $url->setOptions($options);
  }

  /**
   * Returns the share area plugin instance from the view.
   *
   * This method looks in the view's header and footer for a plugin instance of
   * the share area.
   *
   * @param \Drupal\views\ViewExecutable $view
   *   The view object.
   *
   * @return \Drupal\views_share\Plugin\views\area\ShareArea|null
   *   The share area plugin instance, or NULL if not found.
   */
  public static function findPlugin(ViewExecutable $view): ?ShareArea {
    $display = $view->getDisplay();
    foreach (['header', 'footer'] as $area) {
      // Get the plugins for the given area.
      $area_handlers = $display->getHandlers($area);
      if (!empty($area_handlers)) {
        foreach ($area_handlers as $plugin) {
          if ($plugin instanceof ShareArea) {
            return $plugin;
          }
        }
      }
    }

    return NULL;
  }

  /**
   * Removes the _wrapper_format key from the query array.
   *
   * @param array $query
   *   The query array to modify.
   *
   * @return array
   *   The modified query array.
   */
  public static function removeWrapperFormat(array $query): array {
    if (array_key_exists('_wrapper_format', $query)) {
      unset($query['_wrapper_format']);
    }

    return $query;
  }

  /**
   * Builds a shareable URL.
   *
   * @param \Drupal\Core\Url $url
   *   The URL to be built.
   * @param bool $remove_referrer
   *   The remove referrer option.
   *
   * @return \Drupal\Core\Url
   *   The shareable URL.
   */
  public static function buildShareableUrl(Url $url, bool $remove_referrer = FALSE): Url {
    // Remove the views_share_args from the query string.
    $query = $url->getOption('query');
    if (isset($query['views_share_args'])) {
      unset($query['views_share_args']);
    }
    if ($remove_referrer && isset($query['referer'])) {
      unset($query['referer']);
    }
    return $url->setOption('query', $query);
  }

}
