<?php

namespace Drupal\views_slider\Plugin\views\style;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Plugin\views\style\StylePluginBase;

/**
 * Style plugin to render each item in a views_slider.
 *
 * @ingroup views_style_plugins
 *
 * @ViewsStyle(
 *   id = "views_slider",
 *   title = @Translation("Views Slider"),
 *   help = @Translation("Display the results as a views_slider."),
 *   theme = "views_view_views_slider",
 *   display_types = {"normal"}
 * )
 */
class ViewsSlider extends StylePluginBase {

  /**
   * Does the style plugin allow the use of row plugins.
   *
   * @var bool
   */
  protected $usesRowPlugin = TRUE;

  /**
   * Does the style plugin support custom CSS classes for the rows.
   *
   * @var bool
   */
  protected $usesRowClass = TRUE;

  /**
   * Does the style plugin support grouping of rows.
   *
   * @var bool
   */
  protected $usesGrouping = FALSE;

  /**
   * Does the style plugin support adding fields to its output.
   *
   * @var bool
   */
  protected $usesFields = TRUE;

  /**
   * {@inheritdoc}
   */
  protected function defineOptions() {
    $options = parent::defineOptions();

    // General Options.
    $options['row_class_custom'] = ['default' => ''];
    $options['row_class_default'] = ['default' => TRUE];
    $options['views_slider_type'] = ['default' => ''];
    $options['views_slider_heading'] = ['default' => 'default'];
    $options['views_slider_description1'] = ['default' => ''];
    $options['views_slider_description2'] = ['default' => ''];
    $options['views_slider_background_image'] = ['default' => ''];
    $options['views_slider_link1'] = ['default' => 'default'];
    $options['views_slider_link2'] = ['default' => 'default'];

    // Swiper JS configuration options.
    $swiper_defaults = [
      'swiper_direction' => 'horizontal',
      'swiper_loop' => TRUE,
      'swiper_speed' => 800,
      'swiper_space_between' => 0,
      'swiper_slides_per_view' => 1,
      'swiper_centered_slides' => FALSE,
      'swiper_grab_cursor' => TRUE,
      'swiper_effect' => 'coverflow',
      'swiper_navigation' => TRUE,
      'swiper_pagination' => TRUE,
      'swiper_keyboard_enabled' => TRUE,
      'swiper_keyboard_only_in_viewport' => TRUE,
      'swiper_mousewheel_enabled' => TRUE,
      'swiper_mousewheel_sensitivity' => 1,
      'swiper_mousewheel_release_on_edges' => TRUE,
      'swiper_lazy_loading_load_prev_next' => TRUE,
      'swiper_lazy_loading_load_prev_next_amount' => 1,
      'swiper_lazy_loading_load_on_transition_start' => FALSE,
      'swiper_simulate_touch' => TRUE,
      'swiper_autoplay' => 1000,
      'swiper_autoplay_disable_on_interaction' => FALSE,
      'swiper_autoplay_pause_on_mouseenter' => TRUE,
    ];

    foreach ($swiper_defaults as $key => $default) {
      $options[$key] = ['default' => $default];
    }

    // Breakpoints (Mobile, Tablet, Desktop)
    // - Group breakpoint-specific options.
    $breakpoint_defaults = [
      'swiper_breakpoints_320' => 1,
      'swiper_breakpoints_768' => 1,
      'swiper_breakpoints_1200' => 1,
      'swiper_space_between_320' => 0,
      'swiper_space_between_768' => 0,
      'swiper_space_between_1200' => 0,
    ];

    foreach ($breakpoint_defaults as $key => $default) {
      $options[$key] = ['default' => $default];
    }

    return $options;
  }

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state) {
    parent::buildOptionsForm($form, $form_state);

    // Define the common field settings for Slider Heading,
    // Descriptions, and Links.
    $slider_fields = [
      'views_slider_heading' => [
        'title' => 'Slider Heading',
        'description' => 'Select a field to be used as the title of the slider.',
        'prefix' => '<div class="slider-flex-wrapper slider-config-wrapper">',
      ],
      'views_slider_description1' => [
        'title' => 'Slider Description 1',
        'description' => 'Choose a field for the primary description text for the slider item.',
      ],
      'views_slider_description2' => [
        'title' => 'Slider Description 2',
        'description' => 'Select a field for the secondary description for the slider.',
      ],
      'views_slider_background_image' => [
        'title' => 'Slider Background Image',
        'description' => 'Select a field to be used as the background image for the slider.',
      ],
      'views_slider_link1' => [
        'title' => 'Slider Link 1',
        'description' => 'Choose a field for the first link for the slider item.',
      ],
      'views_slider_link2' => [
        'title' => 'Slider Link 2',
        'description' => 'Select a field for a second link option for the slider.',
        'suffix' => '</div>',
      ],
    ];

    // Generate form elements for each slider field.
    foreach ($slider_fields as $field_name => $field) {
      $form[$field_name] = [
        '#type' => 'select',
        '#title' => $this->t('@title', ['@title' => $field['title']]),
        '#options' => ['' => $this->t('None')] + $this->getNonExcludedFields(),
        '#default_value' => $this->options[$field_name],
        '#description' => $this->t('@description', ['@description' => $field['description']]),
        '#prefix' => $field['prefix'] ?? '',
        '#suffix' => $field['suffix'] ?? '',
      ];
    }

    // Swiper Settings Fieldset.
    $form['swiper_settings'] = [
      '#type' => 'fieldset',
      '#title' => $this->t('Swiper Configuration'),
      '#collapsible' => TRUE,
      '#collapsed' => FALSE,
    ];

    // Swiper Direction.
    $form['swiper_settings']['swiper_direction'] = [
      '#type' => 'select',
      '#title' => $this->t('Slider Direction'),
      '#options' => [
        'horizontal' => $this->t('Horizontal'),
        'vertical' => $this->t('Vertical'),
      ],
      '#default_value' => $this->options['swiper_settings']['swiper_direction'],
      '#description' => $this->t('Choose the direction in which the slides will move. You can select "horizontal" for left-right sliding or "vertical" for up-down sliding.'),
    ];

    // Swiper Loop.
    $form['swiper_settings']['swiper_loop'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Loop'),
      '#default_value' => $this->options['swiper_settings']['swiper_loop'],
      '#description' => $this->t('Enable continuous looping of slides, meaning the first slide will appear after the last slide when the user reaches the end.'),
    ];

    // Swiper Speed.
    $form['swiper_settings']['swiper_speed'] = [
      '#type' => 'number',
      '#title' => $this->t('Slide Speed (ms)'),
      '#default_value' => $this->options['swiper_settings']['swiper_speed'],
      '#min' => 0,
      '#description' => $this->t('Set the duration of the slide transition in milliseconds. A lower value means faster transitions.'),
    ];

    // Swiper Space Between Slides.
    $form['swiper_settings']['swiper_space_between'] = [
      '#type' => 'number',
      '#title' => $this->t('Space Between Slides'),
      '#default_value' => $this->options['swiper_settings']['swiper_space_between'],
      '#min' => 0,
      '#description' => $this->t('Set the space between each slide in pixels. Adjust to control the spacing between the slider items.'),
    ];

    // Swiper Slides Per View.
    $form['swiper_settings']['swiper_slides_per_view'] = [
      '#type' => 'number',
      '#title' => $this->t('Slides Per View'),
      '#default_value' => $this->options['swiper_settings']['swiper_slides_per_view'],
      '#min' => 1,
      '#description' => $this->t('Determine how many slides will be displayed at once. This value is particularly useful when you want to show multiple slides at a time.'),
    ];

    // Swiper Centered Slides.
    $form['swiper_settings']['swiper_centered_slides'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Centered Slides'),
      '#default_value' => $this->options['swiper_settings']['swiper_centered_slides'],
      '#description' => $this->t('Enable to center the active slide in the viewport for better focus.'),
    ];

    // Grab Cursor.
    $form['swiper_settings']['swiper_grab_cursor'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Grab Cursor'),
    // Default to TRUE if not set.
      '#default_value' => $this->options['swiper_settings']['swiper_grab_cursor'] ?? TRUE,
      '#description' => $this->t('Enable to show a "grab" cursor when hovering over the swiper. This is useful for indicating that the user can swipe the slider.'),
    ];

    // Swiper Autoplay Delay.
    $form['swiper_settings']['swiper_autoplay'] = [
      '#type' => 'number',
      '#title' => $this->t('Autoplay Delay (ms)'),
      '#default_value' => $this->options['swiper_settings']['swiper_autoplay'],
      '#min' => 0,
      '#description' => $this->t('Set the autoplay delay in milliseconds between each slide transition. This is used when autoplay is enabled.'),
    ];

    // Autoplay Disable on Interaction.
    $form['swiper_settings']['swiper_autoplay_disable_on_interaction'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Disable on Interaction'),
      '#default_value' => $this->options['swiper_settings']['swiper_autoplay_disable_on_interaction'] ?? FALSE,
      '#description' => $this->t('Disable autoplay when the user interacts with the slider (like clicking on a slide). When unchecked, autoplay will continue after user interaction.'),
    ];

    // Autoplay Pause on Mouse Enter.
    $form['swiper_settings']['swiper_autoplay_pause_on_mouseenter'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Pause on Mouse Enter'),
      '#default_value' => $this->options['swiper_settings']['swiper_autoplay_pause_on_mouseenter'] ?? TRUE,
      '#description' => $this->t('Stop autoplay when the mouse hovers over the slider. Enable this option to pause autoplay when the user hovers over the slider to inspect the content.'),
    ];

    // Swiper Effect.
    $form['swiper_settings']['swiper_effect'] = [
      '#type' => 'select',
      '#title' => $this->t('Slide Transition Effect'),
      '#options' => [
        'slide' => $this->t('Slide'),
        'fade' => $this->t('Fade'),
        'cube' => $this->t('Cube'),
        'coverflow' => $this->t('Coverflow'),
        'flip' => $this->t('Flip'),
        'creative' => $this->t('Creative'),
        'cards' => $this->t('Cards'),
      ],
      '#default_value' => $this->options['swiper_settings']['swiper_effect'],
      '#description' => $this->t('Select the transition effect for slides. This controls the animation between slide changes.'),
    ];

    // Swiper Navigation (Arrows)
    $form['swiper_settings']['swiper_navigation'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Navigation Arrows'),
      '#default_value' => $this->options['swiper_settings']['swiper_navigation'],
      '#description' => $this->t('Enable next/previous arrows for manual slide navigation.'),
    ];

    // Swiper Pagination (Dots)
    $form['swiper_settings']['swiper_pagination'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Enable Pagination'),
      '#default_value' => $this->options['swiper_settings']['swiper_pagination'],
      '#description' => $this->t('Enable pagination dots for users to navigate to a specific slide.'),
    ];

    // Pagination Type.
    $form['swiper_settings']['pagination_type'] = [
      '#type' => 'select',
      '#title' => $this->t('Pagination Type'),
      '#options' => [
        'bullets' => $this->t('Bullets'),
        'fraction' => $this->t('Fraction'),
        'progress' => $this->t('Progress'),
        'custom' => $this->t('Custom'),
      ],
      '#default_value' => $this->options['swiper_settings']['pagination_type'],
      '#description' => $this->t('Choose the type of pagination when enabled.'),
      '#states' => [
        'visible' => [
          ':input[name="style_options[swiper_settings][swiper_pagination]"]' => ['checked' => TRUE],
        ],
      ],
    ];

    // Keyboard Control.
    $form['swiper_settings']['swiper_keyboard'] = [
      '#type' => 'details',
      '#title' => $this->t('Keyboard Control'),
      '#open' => FALSE,
      '#description' => $this->t('Configure keyboard control settings for the swiper.'),
      'swiper_keyboard_enabled' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable Keyboard Control'),
        '#default_value' => $this->options['swiper_settings']['swiper_keyboard_enabled'] ?? TRUE,
        '#description' => $this->t('Enable keyboard control for navigating through slides.'),
      ],
      'swiper_keyboard_only_in_viewport' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Only in Viewport'),
        '#default_value' => $this->options['swiper_settings']['swiper_keyboard_only_in_viewport'] ?? TRUE,
        '#description' => $this->t('Enable keyboard control only when the swiper is in the viewport.'),
      ],
    ];

    // Mousewheel Control.
    $form['swiper_settings']['swiper_mousewheel'] = [
      '#type' => 'details',
      '#title' => $this->t('Mousewheel Control'),
      '#open' => FALSE,
      '#description' => $this->t('Configure mousewheel control settings for the swiper.'),
      'swiper_mousewheel_enabled' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Enable Mousewheel Control'),
        '#default_value' => $this->options['swiper_settings']['swiper_mousewheel_enabled'] ?? TRUE,
        '#description' => $this->t('Enable mousewheel or trackpad control for navigating slides.'),
      ],
      'swiper_mousewheel_sensitivity' => [
        '#type' => 'number',
        '#title' => $this->t('Mousewheel Sensitivity'),
        '#default_value' => $this->options['swiper_settings']['swiper_mousewheel_sensitivity'] ?? 1,
        '#min' => 0,
        '#description' => $this->t('Set the sensitivity of mousewheel scrolling for navigating slides.'),
      ],
      'swiper_mousewheel_release_on_edges' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Release on Edges'),
        '#default_value' => $this->options['swiper_settings']['swiper_mousewheel_release_on_edges'] ?? TRUE,
        '#description' => $this->t('Stop mousewheel scrolling when the swiper reaches the edges of the slides.'),
      ],
    ];

    // Simulate Touch (for desktop)
    $form['swiper_settings']['swiper_simulate_touch'] = [
      '#type' => 'checkbox',
      '#title' => $this->t('Simulate Touch for Desktop'),
      '#default_value' => $this->options['swiper_settings']['swiper_simulate_touch'] ?? TRUE,
      '#description' => $this->t('Enable touch interactions on desktop devices to simulate mobile-like behavior.'),
    ];

    $form['swiper_settings']['swiper_lazy_loading'] = [
      '#type' => 'details',
      '#title' => $this->t('Lazy Loading'),
      '#open' => FALSE,
      '#description' => $this->t('Configure lazy loading settings for the swiper images.'),

      // Load Previous and Next Images.
      'swiper_lazy_loading_load_prev_next' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Load Previous and Next Images in Advance'),
        '#default_value' => $this->options['swiper_settings']['swiper_lazy_loading_load_prev_next'] ?? TRUE,
        '#description' => $this->t('Enable to load previous and next images before they are visible.'),
      ],

      // Load Previous and Next Images Amount.
      'swiper_lazy_loading_load_prev_next_amount' => [
        '#type' => 'number',
        '#title' => $this->t('Number of Images to Load in Advance'),
        '#default_value' => $this->options['swiper_settings']['swiper_lazy_loading_load_prev_next_amount'] ?? 1,
        '#min' => 1,
        '#description' => $this->t('The number of previous and next images to load before they are in view.'),
      ],

      // Load Images on Transition Start.
      'swiper_lazy_loading_load_on_transition_start' => [
        '#type' => 'checkbox',
        '#title' => $this->t('Load Images on Transition Start'),
        '#default_value' => $this->options['swiper_settings']['swiper_lazy_loading_load_on_transition_start'] ?? FALSE,
        '#description' => $this->t('Enable to load images as soon as the transition starts, even before the user scrolls to them.'),
      ],
    ];

    // Breakpoints (Responsive Configurations for Mobile, Tablet, Desktop)
    $form['swiper_settings']['breakpoints_320'] = [
      '#type' => 'details',
      '#title' => $this->t('Responsive Settings for 320px Width'),
    // Default open, set to FALSE if you want it collapsed.
      '#open' => FALSE,
      '#description' => $this->t('Configure the number of slides and space between slides for a viewport width of 320px or smaller.'),
      'swiper_breakpoints_320' => [
        '#type' => 'number',
        '#title' => $this->t('Slides Per View for 320px Width'),
        '#default_value' => $this->options['swiper_settings']['breakpoints_320']['swiper_breakpoints_320'],
        '#min' => 1,
        '#description' => $this->t('Number of slides visible when the viewport width is 320px or smaller.'),
      ],
      'swiper_space_between_320' => [
        '#type' => 'number',
        '#title' => $this->t('Space Between Slides for 320px Width'),
        '#default_value' => $this->options['swiper_settings']['breakpoints_320']['swiper_space_between_320'] ?? 0,
        '#min' => 0,
        '#description' => $this->t('Space between slides when the screen width is 320px.'),
      ],
    ];

    $form['swiper_settings']['breakpoints_768'] = [
      '#type' => 'details',
      '#title' => $this->t('Responsive Settings for 768px Width'),
      '#open' => FALSE,
      '#description' => $this->t('Configure the number of slides and space between slides for a viewport width of 768px.'),
      'swiper_breakpoints_768' => [
        '#type' => 'number',
        '#title' => $this->t('Slides Per View for 768px Width'),
        '#default_value' => $this->options['swiper_settings']['breakpoints_768']['swiper_breakpoints_768'],
        '#min' => 1,
        '#description' => $this->t('Number of slides visible when the viewport width is 768px.'),
      ],
      'swiper_space_between_768' => [
        '#type' => 'number',
        '#title' => $this->t('Space Between Slides for 768px Width'),
        '#default_value' => $this->options['swiper_settings']['breakpoints_768']['swiper_space_between_768'] ?? 0,
        '#min' => 0,
        '#description' => $this->t('Space between slides when the screen width is 768px.'),
      ],
    ];

    $form['swiper_settings']['breakpoints_1200'] = [
      '#type' => 'details',
      '#title' => $this->t('Responsive Settings for 1200px Width'),
      '#open' => FALSE,
      '#description' => $this->t('Configure the number of slides and space between slides for a viewport width of 1200px or larger.'),
      'swiper_breakpoints_1200' => [
        '#type' => 'number',
        '#title' => $this->t('Slides Per View for 1200px Width'),
        '#default_value' => $this->options['swiper_settings']['breakpoints_1200']['swiper_breakpoints_1200'],
        '#min' => 1,
        '#description' => $this->t('Number of slides visible when the viewport width is 1200px or larger.'),
      ],
      'swiper_space_between_1200' => [
        '#type' => 'number',
        '#title' => $this->t('Space Between Slides for 1200px Width'),
        '#default_value' => $this->options['swiper_settings']['breakpoints_1200']['swiper_space_between_1200'] ?? 0,
        '#min' => 0,
        '#description' => $this->t('Space between slides when the screen width is 1200px or larger.'),
      ],
    ];

    // Add a library to style the form.
    $form['#attached']['library'] = ['views_slider/form'];
  }

  /**
   * {@inheritdoc}
   */
  public function validateOptionsForm(&$form, FormStateInterface $form_state) {
    // Validate the options form values if necessary.
  }

  /**
   * {@inheritdoc}
   */
  public function submitOptionsForm(&$form, FormStateInterface $form_state) {
    // Submit the form values if necessary.
  }

  /**
   * Get an array of non-excluded fields for the current view.
   *
   * @return array
   *   An array of field names.
   */
  protected function getNonExcludedFields() {
    $fields = [];
    $all_fields = $this->view->display_handler->getHandlers('field');

    foreach ($all_fields as $field_name => $field_handler) {
      if (empty($field_handler->options['exclude'])) {
        $fields[$field_name] = $field_handler->adminLabel();
      }
    }

    return $fields;
  }

}
