# Views String Aggregation

A Drupal module that provides database-specific SQL string aggregation support for Views, enabling efficient concatenation of multiple field values into a single aggregated result.

## Features

- **Database-Specific Implementation**: Supports MySQL/MariaDB `GROUP_CONCAT` and PostgreSQL `STRING_AGG` functions
- **SQL-Level Aggregation**: Performs concatenation at the database level for optimal performance
- **Configurable Separators**: Customize the separator used between concatenated values
- **Sorting Support**: Order concatenated values with MySQL/MariaDB support for custom ordering
- **Views Integration**: Seamlessly extends Drupal Views with minimal configuration
- **Exposed Filter Support**: Works with Views exposed filters and grouping
- **Distinct Values**: Support for distinct value aggregation

## Requirements

- Drupal 10.x or 11.x
- Views module (core)
- Supported databases:
  - MySQL 5.7+ / MariaDB 10.2+
  - PostgreSQL 9.1+

## Installation

### Composer (Recommended)

```
composer require drupal/views_string_aggregation
```

### Manual (Drush)

1. Download and place the module in your `modules/custom/` directory
2. Enable the module: `drush en views_string_aggregation`
3. Clear cache: `drush cr`

### Manual (UI)

1. Download and place the module in your `modules/custom/` directory
2. Enable the module at Admin > Modules

## Usage

### Basic Setup

1. Create or edit a View
2. Enable **"Use aggregation"** in the view's **Advanced** settings
3. Add fields you want to aggregate
4. Set the field's **Aggregation type** to **"String aggregation"**
5. Configure grouping by adding appropriate fields with **"Group results together"**

### Configuration Options

#### Query Settings
When aggregation is enabled, additional options become available:

- **Concatenated values separator**: Character(s) to separate aggregated values (default: `,`)
- **Order concatenated values by**: (MySQL/MariaDB only) Select a field to order concatenated values
- **Order direction**: Ascending or Descending order for concatenated values

#### Field Configuration
Set fields to use string aggregation by selecting **"String aggregation"** as the aggregation type in the field configuration.

### Example Use Cases

#### 1. Node Tags Aggregation
Group nodes by category and show all tags as a comma-separated list:
```
Content Type | Tags
-------------|------------------
Article      | drupal, cms, web
Blog         | personal, tech
```

#### 2. User Roles Summary
Display all roles for users grouped by department:
```
Department | User Roles
-----------|------------------------
IT         | administrator, editor
Marketing  | editor, content creator
```

## Advanced Configuration

### Disable Automatic Query Plugin Extension
To manually control which views use string aggregation:

1. Set `extend_views_query_plugin: false` in configuration
2. Clear cache
3. Manually assign query plugins in `hook_views_data()`:

```php
// mysql
$views_data['table']['base']['query_id'] = 'vsa_views_query_mysql';
// or postgres
$views_data['table']['base']['query_id'] = 'vsa_views_query_pgsql';
```

### Database-Specific Features

#### MySQL/MariaDB
- Supports `ORDER BY` within `GROUP_CONCAT`
- Configurable separator and ordering
- Distinct value aggregation

#### PostgreSQL
- Uses `STRING_AGG` for standard aggregation
- Uses `ARRAY_AGG` with `array_to_string` for distinct values
- Configurable separator

## API Reference

### Hook Implementations

- `hook_views_plugins_query_alter()`: Automatically extends the default Views query plugin
- `hook_config_schema_info_alter()`: Extends Views query schema with custom options

### Query Plugin Classes

- `VsaBase`: Base class extending Drupal's SQL query plugin
- `MySql`: MySQL/MariaDB-specific implementation with `GROUP_CONCAT`
- `PgSql`: PostgreSQL-specific implementation with `STRING_AGG`

### Filter Handlers

- `GroupByString`: Specialized filter for grouped string fields using `HAVING` clauses

## Troubleshooting

### Common Issues

**String aggregation option not appearing**
- Ensure "Use aggregation" is enabled in View advanced settings
- Verify the module is enabled and cache is cleared

**Separator not working**
- Check that the separator is configured in the Query settings
- Ensure aggregation is properly enabled

**Ordering not available**
- Ordering within concatenation is only available for MySQL/MariaDB
- Ensure appropriate database driver is detected

### Performance Considerations

- String aggregation is performed at the SQL level for optimal performance
- Large result sets with many concatenated values may impact query performance
- Consider using `DISTINCT` aggregation when appropriate to reduce redundant data

## Testing

The module includes comprehensive kernel tests covering:
- Basic string aggregation functionality
- Separator configuration
- Exposed filter integration
- Sort order validation
- Database-specific implementations

Run tests with:
```bash
./vendor/bin/phpunit modules/custom/views_string_aggregation/tests/
```

## Credits

This module acknowledges with thanks the work contributed in
<https://www.drupal.org/project/drupal/issues/2902481> upon which this module is based by:

- f2boot
- phjou
- navkaur
- kapilv
- jackson.cooper
- attisan
- simeonkesmev
- achton
- kasperg
- Sir-Arturio
- b.ravanbakhsh

## Contributing

This module follows Drupal coding standards and best practices. Contributions are welcome for:
- Additional database engine support
- Performance optimizations
- Enhanced configuration options
- Bug fixes and improvements

## License

This project is licensed under the GPL-2.0+ license, consistent with Drupal core.

## Support

For issues, feature requests, or questions:
1. Check existing issues in the project repository
2. Review the troubleshooting section above
3. Create a detailed issue report with:
   - Drupal version
   - Database type and version
   - Steps to reproduce
   - Expected vs actual behavior
