<?php

namespace Drupal\views_string_aggregation\Plugin\views\query;

use Drupal\Core\Form\FormStateInterface;
use Drupal\views\Attribute\ViewsQuery;
use Drupal\Core\StringTranslation\TranslatableMarkup;

/**
 * Extended SQL query plugin that adds SQLite GROUP_CONCAT aggregation support.
 *
 * @ingroup views_query_plugins
 */
#[ViewsQuery(
  id: 'vsa_views_query_sqlite',
  no_ui: false,
  title: new TranslatableMarkup('SQL Query with SQLite GROUP_CONCAT'),
  help: new TranslatableMarkup('Query will be generated and run using the Drupal database API with SQLite GROUP_CONCAT support.')
)]
class Sqlite extends VsaBase
{

  /**
   * {@inheritdoc}
   */
  public function buildOptionsForm(&$form, FormStateInterface $form_state)
  {
    parent::buildOptionsForm($form, $form_state);
    // Add some SQLite specific options if use aggregation is enabled on the view.
    if (!empty($form['vsa_separator'])) {
      $field_options = $this->getFieldOptions();
      if (!empty($field_options)) {
        $form['vsa_order_by'] = [
          '#type' => 'select',
          '#title' => $this->t('Order concatenated values by'),
          '#default_value' => $this->options['vsa_order_by'] ?? '',
          '#options' => ['' => $this->t('- Default -')] + $field_options,
          '#description' => $this->t('Select a field to order the string aggregated and concatenated values by.'),
        ];
        $form['vsa_order_direction'] = [
          '#type' => 'select',
          '#title' => $this->t('Concatenated values order direction'),
          '#default_value' => $this->options['vsa_order_direction'] ?? 'ASC',
          '#options' => [
            'ASC' => $this->t('Ascending'),
            'DESC' => $this->t('Descending'),
          ],
          '#description' => $this->t('Select the order direction for the selected field.'),
        ];
      }
    }
  }

  /**
   * String aggregation method callback for SQLite GROUP_CONCAT syntax.
   */
  public function vsaAggregationMethodSimple($group_type, $field): string
  {
    return 'GROUP_CONCAT(' . $field . ', ' . $this->getSeparator() . $this->getOrderBy() . ')';
  }

  /**
   * String aggregation method callback for SQLite DISTINCT GROUP_CONCAT syntax.
   */
  public function vsaAggregationMethodDistinct($group_type, $field): string
  {
    // SQLite does not yet support a custom separator with DISTINCT.
    // but we can still support ordering.
    return 'GROUP_CONCAT(DISTINCT ' . $field . $this->getOrderBy() . ')';
  }

}
